package at.tugraz.genome.maspectras.parser.spectrummill;

import java.io.BufferedInputStream;
import java.io.FileInputStream;
import java.util.Collections;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.StringTokenizer;
import java.util.Vector;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import at.tugraz.genome.maspectras.GlobalConstants;
import at.tugraz.genome.maspectras.parser.XMLFileLoader;
import at.tugraz.genome.maspectras.parser.exceptions.SpectrummillParserException;
import at.tugraz.genome.maspectras.parser.spectrummill.vos.ChemicalFormulaVO;
import at.tugraz.genome.maspectras.parser.spectrummill.vos.SmChemicalElementVO;
import at.tugraz.genome.maspectras.parser.spectrummill.vos.SmIsotopeVO;
import at.tugraz.genome.maspectras.parser.spectrummill.vos.SmModificationVO;
import at.tugraz.genome.maspectras.utils.Calculator;
import at.tugraz.genome.voutils.GeneralComparator;

public class ElementConfigParser
{
  /** absolute path to the SM config file */
  protected String configFile_;
  /** lookup for the chemical elements used */
  protected Hashtable<String,SmChemicalElementVO> chemElements;
  /** lookup for the parsed possible modifications */
  protected Hashtable<String,SmModificationVO> modifications;
  
//  protected Vector<Double> probabilities;
//  protected Vector<Double> normalizedValues;

  
  /**
   * @param the absolute path to the SM config file
   */
  public ElementConfigParser(String configFile)
  {
    this.configFile_ = configFile;
    this.chemElements = new Hashtable<String,SmChemicalElementVO>();
    this.modifications = new Hashtable<String,SmModificationVO>();
  }
  
  /**
   * starts the parsing of the SM configuration files
   * @throws SpectrummillParserException
   */
  public void parse() throws SpectrummillParserException{
    try {
      BufferedInputStream in = new BufferedInputStream(new FileInputStream(configFile_));
      XMLFileLoader loader = new XMLFileLoader(in);
      Document doc = loader.getDocument();
      Element rootElement = doc.getDocumentElement();
      if (!rootElement.getNodeName().equalsIgnoreCase(GlobalConstants.SM_CONFIG_XML_ROOT_ELEMENT)) 
        throw new SpectrummillParserException("The root element of a smconfig.xml should be "+GlobalConstants.SM_CONFIG_XML_ROOT_ELEMENT+"!! And not "+rootElement.getNodeName());
      NodeList configurations = rootElement.getChildNodes();
      int nrOfConfigNodes = 0;
      for (int i=0; i!=configurations.getLength();i++){
        Node configNode = configurations.item(i);
        if (configNode.getNodeName().equalsIgnoreCase(GlobalConstants.SM_CONFIG_XML_CONF_ELEMENT)){
          this.parseConfig(configNode);
          nrOfConfigNodes++;
        }
      }
      if (nrOfConfigNodes!=1) throw new SpectrummillParserException("The actual version supports only one \""+GlobalConstants.SM_CONFIG_XML_CONF_ELEMENT+"\" element");
    } catch (java.io.IOException io) {
        throw new SpectrummillParserException("The access to the file could not be established!");
        //      sequence = null;
    }
  }
  
  /**
   * Parses the tage named config (normally here everything is stored)
   * @param configNode the config node
   * @throws SpectrummillParserException
   */
  protected void parseConfig(Node configNode)throws SpectrummillParserException{
    NodeList nl = configNode.getChildNodes();
    int nrOfElementsNodes = 0;
    for (int i=0; i!=nl.getLength(); i++){
      Node node = nl.item(i);
      if (node.getNodeName().equalsIgnoreCase(GlobalConstants.SM_CONFIG_XML_ELEMENTS)){
        Vector<SmChemicalElementVO> chemicalElements = this.parseElementsNode(node);
        for (SmChemicalElementVO elementVO: chemicalElements){
          chemElements.put(elementVO.getChemicalSymbol(),elementVO);
        }
        nrOfElementsNodes++;
      }
    }
    if (nrOfElementsNodes!=1) throw new SpectrummillParserException("The element "+GlobalConstants.SM_CONFIG_XML_CONF_ELEMENT+"must have exactly one \""+GlobalConstants.SM_CONFIG_XML_ELEMENTS+"\" element");
  }
  
  /**
   * parses the contents of the tag elements; here the used chemical elements are stored
   * @param elementsNode elements node
   * @return the VOs for the chemical elements
   * @throws SpectrummillParserException
   */
  protected Vector<SmChemicalElementVO> parseElementsNode (Node elementsNode)throws SpectrummillParserException{
    Vector<SmChemicalElementVO> elements = new Vector<SmChemicalElementVO>();
    NodeList nl = elementsNode.getChildNodes();
    for (int i=0; i!=nl.getLength();i++){
      Node node = nl.item(i);
      if (node.getNodeName().equalsIgnoreCase(GlobalConstants.SM_CONFIG_XML_ELEMENT)){
        elements.add(this.parseElementNode(node));
      }
    }
    return elements;
  }


  /**
   * parses the contents of the tag element; here the mass values for a single element are stroes
   * @param elementNode element node
   * @return the VO for the chemical element
   * @throws SpectrummillParserException
   */
  protected SmChemicalElementVO parseElementNode (Node elementNode) throws SpectrummillParserException{
    Element element = (Element) elementNode;
    String symbol = element.getAttribute(GlobalConstants.SM_CONFIG_XML_ELEMENT_ATTR_SYMBOL);
    if (symbol==null||symbol.length()<1) throw new SpectrummillParserException("The element "+GlobalConstants.SM_CONFIG_XML_ELEMENT+"must have one attribute named \""+GlobalConstants.SM_CONFIG_XML_ELEMENT_ATTR_SYMBOL);
    NodeList nl =  element.getChildNodes();
    Vector<SmIsotopeVO> isotopes = new Vector<SmIsotopeVO>();
    for (int i=0; i!=nl.getLength();i++){
      Node node = nl.item(i);
      if (node.getNodeName().equalsIgnoreCase(GlobalConstants.SM_CONFIG_XML_ELEMENT_ISOTOPE)){
        Element isotope = (Element) node;
        String massString = isotope.getAttribute(GlobalConstants.SM_CONFIG_XML_ELEMENT_ISOTOPE_ATTR_MASS);
        String abundanceString = isotope.getAttribute(GlobalConstants.SM_CONFIG_XML_ELEMENT_ISOTOPE_ATTR_ABUNDANCE);
        if (massString==null||massString.length()<1) throw new SpectrummillParserException("The element "+GlobalConstants.SM_CONFIG_XML_ELEMENT_ISOTOPE+"must have one attribute named \""+GlobalConstants.SM_CONFIG_XML_ELEMENT_ISOTOPE_ATTR_MASS);
        if (abundanceString==null||abundanceString.length()<1) throw new SpectrummillParserException("The element "+GlobalConstants.SM_CONFIG_XML_ELEMENT_ISOTOPE+"must have one attribute named \""+GlobalConstants.SM_CONFIG_XML_ELEMENT_ISOTOPE_ATTR_ABUNDANCE);
        double mass;
        double abundance;
        try{
          mass = Double.parseDouble(massString);
        }catch (NumberFormatException nfx){
          throw new SpectrummillParserException("The value of the attriubte "+GlobalConstants.SM_CONFIG_XML_ELEMENT_ISOTOPE_ATTR_MASS+" at element named "+symbol+" must be double format");
        }
        try{
          abundance = Double.parseDouble(abundanceString);
        }catch (NumberFormatException nfx){
          throw new SpectrummillParserException("The value of the attriubte "+GlobalConstants.SM_CONFIG_XML_ELEMENT_ISOTOPE_ATTR_ABUNDANCE+" at element named "+symbol+" must be double format");
        }
        isotopes.add(new SmIsotopeVO(mass,abundance));
      }
    }
    if (isotopes==null||isotopes.size()<1) throw new SpectrummillParserException("The element "+GlobalConstants.SM_CONFIG_XML_ELEMENT+"must have at least one \""+GlobalConstants.SM_CONFIG_XML_ELEMENT_ISOTOPE+"\" element");
    return new SmChemicalElementVO(symbol,isotopes);
  }

  public Vector<Vector<Double>> calculateChemicalFormulaIntensityDistribution(String chemicalFormula, int lastIsotope,boolean nh4Correction) throws SpectrummillParserException{
    Vector massesAndElements = SmModificationVO.decodeFormula(chemicalFormula, chemElements);
    Vector<SmChemicalElementVO> allElements = (Vector<SmChemicalElementVO>) massesAndElements.get(1);
    if (nh4Correction){
      SmChemicalElementVO chemElement = chemElements.get("N");
      allElements.add(chemElement);
      chemElement = chemElements.get("H");
      for (int i=0; i!=4;i++){
        allElements.add(chemElement);
      }
    }
    Hashtable<String,Vector<SmChemicalElementVO>> elementHash = new Hashtable<String,Vector<SmChemicalElementVO>>();
    for (SmChemicalElementVO element: allElements){
      Vector<SmChemicalElementVO> elements = new Vector<SmChemicalElementVO>();
      if (elementHash.containsKey(element.getChemicalSymbol())){
        elements = elementHash.get(element.getChemicalSymbol());
      }
      elements.add(element);
      elementHash.put(element.getChemicalSymbol(), elements);
    }
    Hashtable<String,Integer> negativeProbs = new Hashtable<String,Integer>();
    for (String key : elementHash.keySet()){
      SmChemicalElementVO vo = elementHash.get(key).get(0);
      int highestProbPosition = 0;
      double highestProb = 0f;
      for (int i=0;i!=vo.getProbabilities().size();i++){
        double prob = vo.getProbabilities().get(i);
        if (prob>highestProb){
          highestProb = prob;
          highestProbPosition = i;
        }
      }
      if (highestProbPosition>0)
        negativeProbs.put(key, highestProbPosition*elementHash.get(key).size());
    }
    return this.calculateProbabilitiesFromElements(elementHash, lastIsotope, negativeProbs);
  }
  
  public Double calculateTheoreticalMass(String chemicalFormula,boolean nh4Correction) throws SpectrummillParserException{
    Vector massesAndElements = SmModificationVO.decodeFormula(chemicalFormula, chemElements);
    Vector<SmChemicalElementVO> allElements = (Vector<SmChemicalElementVO>) massesAndElements.get(1);
    if (nh4Correction){
      SmChemicalElementVO chemElement = chemElements.get("N");
      allElements.add(chemElement);
      chemElement = chemElements.get("H");
      for (int i=0; i!=4;i++){
        allElements.add(chemElement);
      }
    }
    Double totalMass = 0d;
    for (SmChemicalElementVO element : allElements){
      totalMass += element.getMonoMass();
    }
    return totalMass;
  }
  
  protected Vector<Double> calculateProbabilitiesFromElements( Hashtable<String,Vector<SmChemicalElementVO>> elementHash, int lastIsotope){
    Vector<Vector<Double>> normalized = calculateProbabilitiesFromElements(elementHash, lastIsotope, new Hashtable<String,Integer>());
    //in the old implementation, it always requested a positive distribution, thus I return always the position 0 (1 would be the negative distribution)
    return normalized.get(0);
  }
  
  protected Vector<Vector<Double>> calculateProbabilitiesFromElements( Hashtable<String,Vector<SmChemicalElementVO>> elementHash, int lastIsotope,
      Hashtable<String,Integer> negativeProbs){
//    this.probabilities = new Vector<Double>();
    Vector<Vector<Double>> normalized = new Vector<Vector<Double>>();
    Enumeration keys = elementHash.keys();
    int nrOfPlusMValues = 0;
    Vector<SmChemicalElementVO> elements;
    while (keys.hasMoreElements()){
      String symbol = (String)keys.nextElement();
      elements = elementHash.get(symbol);
      nrOfPlusMValues += elements.size()*(elements.get(0).getProbabilities().size()-1);
    }
    
    double totalProbability = 0;
    int startValue = 0;
    if (negativeProbs!=null)
      for (Integer addValue : negativeProbs.values()) startValue+=addValue;
    double probability = -1;
    Vector<Double> probabilities = new Vector<Double>();
    for (int i=startValue; (i!=nrOfPlusMValues&&i!=(lastIsotope+startValue));i++){
      probability = this.calculateProbabilitiesForIsotope(elementHash, i, nrOfPlusMValues);
      probabilities.add(new Double(probability));
      totalProbability+=probability;
    }
    Vector<Double> normalizedValues = this.calculateNormalizedValues(probabilities);
    normalized.add(normalizedValues);
    if (negativeProbs.size()>0){
      probabilities = new Vector<Double>();
      for (int i=startValue; i!=-1&&i!=(startValue-lastIsotope);i--){
        probability = this.calculateProbabilitiesForIsotope(elementHash, i, nrOfPlusMValues);
        probabilities.add(new Double(probability));
        totalProbability+=probability;
      }
      normalizedValues = this.calculateNormalizedValues(probabilities);
      normalized.add(normalizedValues);
    }
//    if (negDistribution){
//      for (int i=startValue; i!=-1&&i!=(startValue-lastIsotope);i--){
//        probability = this.calculateProbabilitiesForIsotope(elementHash, i, nrOfPlusMValues);
//        this.probabilities.add(new Double(probability));
//        totalProbability+=probability;
//      }
//    }else{
//      for (int i=startValue; (i!=nrOfPlusMValues&&i!=(lastIsotope+startValue));i++){
//        probability = this.calculateProbabilitiesForIsotope(elementHash, i, nrOfPlusMValues);
//        this.probabilities.add(new Double(probability));
//        totalProbability+=probability;
//      }
//    }
    return normalized;
  }
  
  private Vector<Double> calculateNormalizedValues(Vector<Double> probabilities){
    double highestValue = 0;
    Vector<Double> normalizedValues = new Vector<Double>();
    if (probabilities!=null&&probabilities.size()>0){
      for (Double probability: probabilities){
        if (probability>highestValue) highestValue = probability; 
      }
      for (Double probability: probabilities){
        normalizedValues.add(new Double(probability/highestValue));
      }
    }
    return normalizedValues;
  }
  
  public double calculateProbabilitiesForIsotope(Hashtable<String,Vector<SmChemicalElementVO>> elementHash, int isotopicMassShift, int nrOfPlusMValues){
    double probability = 0;
    Enumeration<String> keys = elementHash.keys();
    Vector<String> allPossibleElementValues = new Vector<String>();
    while (keys.hasMoreElements()){
      String key = keys.nextElement();
      Vector<SmChemicalElementVO> elements = elementHash.get(key);
      Vector<String> possibleElementValues = this.getPossibleElementValues(elements,isotopicMassShift, nrOfPlusMValues);
      allPossibleElementValues.addAll(possibleElementValues);
    }
//    if (isotopicMassShift==2){
//      for (int i=0;i!=allPossibleElementValues.size();i++){
//        System.out.println(allPossibleElementValues.get(i));
//      }
//    }
/*    System.out.println("!!!!!!!!!!!!!!!!!!!!!!!!!!");*/
////    long time = System.currentTimeMillis();
    Vector<String> allPossibleCombinations = this.getPossibleCombinations(elementHash,allPossibleElementValues, isotopicMassShift);
//    System.out.println("Consumed time1: "+(System.currentTimeMillis()-time)/1000+"sec");
////    time = System.currentTimeMillis();
    long totalNumberOfValues = 0;
////    System.out.println(allPossibleCombinations.size());
    for (int i=0;i!=allPossibleCombinations.size();i++){
      String combination = allPossibleCombinations.get(i);
      double occurence = this.calculateOccurence(elementHash, combination);
//      System.out.println(isotopicMassShift+": "+combination+" "+"("+occurence+")");
      double probabilityValue = this.getProbabilityValue(elementHash, combination, occurence);
      probability += probabilityValue;
//      System.out.println(combination+" ; "+occurence+" ; "+probabilityValue);
      totalNumberOfValues+=occurence;
      
//      System.out.println(i+": "+allPossibleCombinations.get(i)+": "+occurence);
    }
    
    if (isotopicMassShift==0){
      probability += this.getProbabilityValue(elementHash, "", 1);
    }
////    System.out.println("totalNumberOfValues: "+totalNumberOfValues);
////    System.out.println("Nr: of combinations: "+allPossibleCombinations.size());
////    System.out.println("Consumed time: "+(System.currentTimeMillis()-time)/1000+"sec");
    return probability;
  }
  

  private Vector<String> getPossibleElementValues(Vector<SmChemicalElementVO> elements,int isotopicMassShift, int nrOfPlusMValues){
    Vector<String> possibleElementValues = new Vector<String>();
    int nrOfElements = elements.size();
    SmChemicalElementVO elementVO = elements.get(0);
    int nrOfIsotopicPositions = elementVO.getProbabilities().size()-1;
    for (int i=0; i!= nrOfElements;i++){
      for (int j=0; j!=nrOfIsotopicPositions;j++){
        if (((j+1)*(i+1)<=isotopicMassShift)&&(nrOfPlusMValues-(nrOfElements)*(nrOfIsotopicPositions)+(j+1)*(i+1))>=isotopicMassShift){
          possibleElementValues.add(elementVO.getChemicalSymbol()+"_"+(i+1)+"_"+(j+1));
        }
      }
    }
    return possibleElementValues; 
  }
  
  private Vector<String> getPossibleCombinations(Hashtable<String,Vector<SmChemicalElementVO>> elementHash,Vector<String> possibleElementValues,int isotopicMassShift){
    Vector<String> possibleCombinationValues = new Vector<String>();
    for (int i=0;i!=possibleElementValues.size();i++){
      String elementIsotope = possibleElementValues.get(i);
      int usedIsotopes = this.getAmountOfIsotopes(elementIsotope);
      if (usedIsotopes==isotopicMassShift){
        possibleCombinationValues.add(elementIsotope);
      }else{
        Vector<String> foundCombinations = this.getCombinatoryValues(elementHash,possibleElementValues, isotopicMassShift, usedIsotopes,i+1,elementIsotope);
        for (int j=0;j!=foundCombinations.size();j++){
          possibleCombinationValues.add(elementIsotope+","+foundCombinations.get(j));
        }
      }
    }
    return possibleCombinationValues;
  }

  private double calculateOccurence(Hashtable<String,Vector<SmChemicalElementVO>> elementHash, String combination){
    double occurence = 1;
    StringTokenizer tokenizer = new StringTokenizer(combination,",");
    Hashtable<String,Long> usedPositions = new Hashtable<String,Long>();
    Enumeration<String> keys = elementHash.keys();
    while (keys.hasMoreElements()){
      String key = keys.nextElement();
      usedPositions.put(key, new Long(0));
    }
    while (tokenizer.hasMoreTokens()){
      StringTokenizer tokenizer2 = new StringTokenizer(tokenizer.nextToken(),"_");
      String symbol = tokenizer2.nextToken();
      long amountOfIsotopePositions= Long.parseLong(tokenizer2.nextToken());
//      int carryingIsotopes = Integer.parseInt(tokenizer2.nextToken());
      long totalAmountofThatElement = elementHash.get(symbol).size();
      long usedPosition=usedPositions.get(symbol);
      occurence = occurence*Calculator.calculateNOverK(totalAmountofThatElement-usedPosition, amountOfIsotopePositions);
      usedPosition++;
      usedPositions.put(symbol, new Long(usedPosition));
    }
    return occurence;
  }
  
  private double getProbabilityValue(Hashtable<String,Vector<SmChemicalElementVO>> elementHash, String combination, double occurence){
    double probability=1;
    probability = probability*occurence;
    StringTokenizer tokenizer = new StringTokenizer(combination,",");
    Hashtable<String,Vector<String>> changedIsotopes = new  Hashtable<String,Vector<String>>();

    while (tokenizer.hasMoreTokens()){
      String token = tokenizer.nextToken();
      Vector<String> isotopesToChange = new Vector<String>();
      String symbol = this.getIsotopeSymbol(token);
      if (changedIsotopes.containsKey(symbol)){
        isotopesToChange = changedIsotopes.get(symbol);
      }
      isotopesToChange.add(token);
      changedIsotopes.put(symbol, isotopesToChange);
   }
    
    
   Enumeration<String> keys = elementHash.keys();
    while (keys.hasMoreElements()){
      String key = keys.nextElement();
      Vector<SmChemicalElementVO> elements = elementHash.get(key);
      SmChemicalElementVO element = elements.get(0);
      int amountOfElements = elements.size();
      if (changedIsotopes.containsKey(key)){
        Vector<String> isotopesToChange = changedIsotopes.get(key);
        for (String isotopeToChange: isotopesToChange){
          tokenizer = new StringTokenizer(isotopeToChange,"_");
          tokenizer.nextToken();
          int amount = Integer.parseInt(tokenizer.nextToken());
          int isotope = Integer.parseInt(tokenizer.nextToken());
          for (int i=0;i!=amount;i++){
            probability = probability*element.getProbabilities().get(isotope);
            amountOfElements--;
          }
        }
      }
      for (int i=0; i!=amountOfElements;i++){
        probability = probability*element.getProbabilities().get(0);
      }
    }    
    
    return probability;
  }

  private int getAmountOfIsotopes(String elementIsotope){
    StringTokenizer tokenizer = new StringTokenizer(elementIsotope,"_");
    tokenizer.nextToken();
    int amountOfElements = Integer.parseInt(tokenizer.nextToken());
    int isotopes = Integer.parseInt(tokenizer.nextToken());
    return (amountOfElements*isotopes);
  }
  

  private Vector<String> getCombinatoryValues(Hashtable<String,Vector<SmChemicalElementVO>> elementHash,Vector<String> possibleElementValues,int isotopicMassShift,int usedIsotopes,int startValue,String usedIsotopeString)/* throws SpectrummillParserException*/{
    Vector<String> possibleCombinations = new Vector<String>();
    for (int i=startValue;i<possibleElementValues.size();i++){
      String elementIsotope = possibleElementValues.get(i);
      if (!this.hasSameElementIsotopeConfiguration(elementIsotope,usedIsotopeString)){
        int elementIsotopes = this.getAmountOfIsotopes(elementIsotope);
        if ((usedIsotopes+elementIsotopes)==isotopicMassShift){
          if (this.checkEnoughElementsLeft(elementHash, elementIsotope, usedIsotopeString)){
            possibleCombinations.add(elementIsotope);
          }
        }else if ((usedIsotopes+elementIsotopes)<isotopicMassShift){
          String alreadyUsed = usedIsotopeString+","+elementIsotope;
          int newIsotopeNumber = usedIsotopes+elementIsotopes;
          if (this.checkEnoughElementsLeft(elementHash, elementIsotope, usedIsotopeString)){
            Vector<String> returnedCombinations = this.getCombinatoryValues(elementHash, possibleElementValues, isotopicMassShift, newIsotopeNumber, i+1, alreadyUsed);
            for (String combination: returnedCombinations){
              possibleCombinations.add(elementIsotope+","+combination);
            }
          }  
        }
      }
    }
    return possibleCombinations;
  }
  
  private boolean checkEnoughElementsLeft(Hashtable<String,Vector<SmChemicalElementVO>> elementHash, String elementIsotope, String usedIsotopeString){
    String symbol = this.getIsotopeSymbol(elementIsotope);
    int possibleAmount = elementHash.get(symbol).size();
    int alreadyAdded = this.getAmountOfAlreadyAdded(symbol,usedIsotopeString+","+elementIsotope);
    if (alreadyAdded>/****=*/possibleAmount)
      return false;
    else
      return true;
  }
  
  private String getIsotopeSymbol(String elementIsotope){
    StringTokenizer tokenizer = new StringTokenizer(elementIsotope,"_");
    return tokenizer.nextToken();
  }

  private boolean hasSameElementIsotopeConfiguration(String elementIsotope,String usedIsotopeString){
    boolean hasTheSame=false;
    String symbol = this.getIsotopeSymbol(elementIsotope);
    int configuration = this.getIsotopeConfiguration(elementIsotope);
    StringTokenizer tokenizer = new StringTokenizer(usedIsotopeString,",");
    while (tokenizer.hasMoreTokens()){
      String token = tokenizer.nextToken();
      String currentSymbol = this.getIsotopeSymbol(token);
      int currentConfiguration = this.getIsotopeConfiguration(token);
      if (symbol.equalsIgnoreCase(currentSymbol)&&configuration==currentConfiguration) hasTheSame=true;
    }    
    return hasTheSame;
  }
  
  private int getIsotopeConfiguration(String elementIsotope){
    StringTokenizer tokenizer = new StringTokenizer(elementIsotope,"_");
    tokenizer.nextToken();
    tokenizer.nextToken();
    return Integer.parseInt(tokenizer.nextToken());
  }
  
  private int getAmountOfAlreadyAdded(String symbol,String usedIsotopeString){
    int amount=0;
    StringTokenizer tokenizer = new StringTokenizer(usedIsotopeString,",");
    while (tokenizer.hasMoreTokens()){
      String token = tokenizer.nextToken();
      String currentSymbol = this.getIsotopeSymbol(token);
      if (symbol.equalsIgnoreCase(currentSymbol)){
        StringTokenizer tokenizer2 = new StringTokenizer(token,"_");
        tokenizer2.nextToken();
        amount += Integer.parseInt(tokenizer2.nextToken());
      }
    }
    return amount;
  }

  public Vector<ChemicalFormulaVO> calculatePossibleElementCompositions(float mass, float tolerance){
    float lowerThreshold = mass-tolerance;
    float upperThreshold = mass+tolerance;
    Vector<SmChemicalElementVO> elements = new Vector<SmChemicalElementVO>();   
    for (String key : chemElements.keySet()){
      elements.add(chemElements.get(key));
    }
    Collections.sort(elements, new GeneralComparator("at.tugraz.genome.maspectras.parser.spectrummill.vos.SmChemicalElementVO", "getMonoMass", "java.lang.Double"));
    ChemicalFormulaVO formulaVO = new ChemicalFormulaVO();
    Vector<ChemicalFormulaVO> theoreticallyPossibleFormulas = new Vector<ChemicalFormulaVO>();
//    for (int i=0; i!=elements.size();i++){
//      SmChemicalElementVO element = elements.get(i);
//      if (i==0){
        theoreticallyPossibleFormulas.addAll(this.fillFormulaWithOneElement(formulaVO, elements.get(0),lowerThreshold,upperThreshold));
        theoreticallyPossibleFormulas.addAll(this.calculatePossibleCompositions(elements, formulaVO, lowerThreshold, upperThreshold));
//        System.out.println("afterProcess: "+formulaVO.getChemicalFormula());
//      }
//    }
     // this removes the very inprobable ones
    Vector<ChemicalFormulaVO> theoreticallyProbableFormulas = new Vector<ChemicalFormulaVO>();
    for (ChemicalFormulaVO finalFormulaVO : theoreticallyPossibleFormulas){
      if (finalFormulaVO.isProbable()){
        theoreticallyProbableFormulas.add(finalFormulaVO);
      }
    }
    return theoreticallyProbableFormulas;
  }
  
  private Vector<ChemicalFormulaVO> calculatePossibleCompositions(Vector<SmChemicalElementVO> elements,ChemicalFormulaVO formulaVO,float lowerThreshold, float upperThreshold){
    Vector<ChemicalFormulaVO> theoreticallyPossibleFormulas = new Vector<ChemicalFormulaVO>();
    Vector<SmChemicalElementVO> myElements = new Vector<SmChemicalElementVO>(elements);
    SmChemicalElementVO lastElement = myElements.get(0);
    myElements.remove(0);
    if (myElements.size()>0){
      SmChemicalElementVO currentElement = myElements.get(0);
      int nrOfElementsToRemove = (int)(currentElement.getMonoMass()/lastElement.getMonoMass())+1;
      formulaVO.removeElement(lastElement, nrOfElementsToRemove);
//      theoreticallyPossibleFormulas.addAll(this.fillFormulaWithOneElement(formulaVO, currentElement,lowerThreshold,upperThreshold));
//      theoreticallyPossibleFormulas.addAll(this.calculatePossibleCompositions(myElements, formulaVO, lowerThreshold, upperThreshold));
      for (int i=formulaVO.getNumberOfItemsOfOneElement(lastElement);i!=-1;i--){
        for (SmChemicalElementVO removableElement: myElements){
          formulaVO.removeAllOfOneElement(removableElement);
        }
        theoreticallyPossibleFormulas.addAll(this.fillFormulaWithOneElement(formulaVO, currentElement,lowerThreshold,upperThreshold));
        theoreticallyPossibleFormulas.addAll(this.calculatePossibleCompositions(myElements, formulaVO, lowerThreshold, upperThreshold));

        formulaVO.removeElement(lastElement);
//        Vector<ChemicalFormulaVO> theoreticallyPossibleFormulas2 = this.fillFormulaWithOneElement(formulaVO, currentElement,lowerThreshold,upperThreshold);
        
//        System.out.println(i+" "+formulaVO.getChemicalFormula()+" "+formulaVO.getMassOfAddedElements()+" ");
      }
    }
    return theoreticallyPossibleFormulas;
  }
  
  private Vector<ChemicalFormulaVO> fillFormulaWithOneElement(ChemicalFormulaVO formulaVO,SmChemicalElementVO element,float lowerThreshold, float upperThreshold){
    Vector<ChemicalFormulaVO> theoreticallyPossibleFormulas = new Vector<ChemicalFormulaVO>();
    while (formulaVO.getMassOfAddedElements()<upperThreshold){
      formulaVO.addElement(element);
      if (formulaVO.getMassOfAddedElements()>lowerThreshold&&formulaVO.getMassOfAddedElements()<upperThreshold)
        theoreticallyPossibleFormulas.add(new ChemicalFormulaVO(formulaVO) );
    }
    return theoreticallyPossibleFormulas;
  }
  
  public boolean isElementAvailable(String chemElement){
    return (this.chemElements.containsKey(chemElement));
  }
  
  public SmChemicalElementVO getElementDetails(String chemElement){
    if (isElementAvailable(chemElement))return this.chemElements.get(chemElement);
    else return null;
  }
}
