package at.tugraz.genome.maspectras.parser.spectrummill.vos;

import java.util.Hashtable;
import java.util.StringTokenizer;
import java.util.Vector;

import at.tugraz.genome.maspectras.msResultsAnalysis.vos.MassreferenceVO;
import at.tugraz.genome.maspectras.parser.exceptions.SpectrummillParserException;
import at.tugraz.genome.maspectras.GlobalConstants;

/**
 * Holds the mass values for the modifications
 * @author Juergen Hartler
 *
 */
public class SmModificationVO
{
  /** SM identifier for the modification */
  private String id;
  /** the name of the modification*/
  private String name;
  /** the chemical formula of the modification*/
  private String formula;
  /** type of the modification */
  private String type;
  /** amino acids of the modifications */
  private String modificationSite;
  /** the ids for the cyclic modifications (ids to other modifications) */
  private Vector<String> cyclicModificationIds;
  
  /**
   * @param id SM identifier for the modification
   * @param name name of the modification
   * @param type type of the modification
   * @param modificationSite amino acids of the modifications
   */
  private SmModificationVO(String id, String name, String type, String modificationSite){
    this.id = id;
    this.name = name;
    this.type = type;
    this.modificationSite = modificationSite;
  }
  
  /**
   * @param id SM identifier for the modification
   * @param name name of the modification
   * @param type type of the modification
   * @param modificationSite amino acids of the modifications
   * @param formula chemical formula of the modification
   */
  public SmModificationVO(String id, String name, String type, String modificationSite, String formula){
    this (id, name, type, modificationSite);
    this.formula = formula;
  }
  
  /**
   * @param id SM identifier for the modification
   * @param name name of the modification
   * @param type type of the modification
   * @param modificationSite amino acids of the modifications
   * @param cyclicModificationIds ids for the cyclic modifications (ids to other modifications)
   */
  public SmModificationVO(String id, String name, String type, String modificationSite, Vector<String> cyclicModificationIds){
    this (id, name, type, modificationSite);
    this.cyclicModificationIds = cyclicModificationIds;
  }

  /**
   * @return SM identifier for the modification
   */
  public String getId()
  {
    return id;
  }
  
  /**
   * @param otherModifications all the modifications
   * @param elements the chemical elements to calculate the modification values
   * @return the monoisotopic mass reference VOs
   * @throws SpectrummillParserException
   */
  public Vector<MassreferenceVO> getMonoisotopicMassreferences(Hashtable<String,SmModificationVO> otherModifications, Hashtable<String,SmChemicalElementVO> elements) throws SpectrummillParserException{
    Vector<MassreferenceVO> refs = new Vector<MassreferenceVO>();
    if (this.type.equalsIgnoreCase(GlobalConstants.SM_CONFIG_XML_MODIFICATION_TYPE_VALUE_CYCLIC)){
      for (String id : this.cyclicModificationIds){
        if (!otherModifications.containsKey(id)) throw new SpectrummillParserException("The cyclic modification "+this.id+"references to the modification"+id+" which does not exist");
        Vector<MassreferenceVO> cyclicRefs = otherModifications.get(id).getMonoisotopicMassreferences(otherModifications,elements);
        for (MassreferenceVO refVO : cyclicRefs) refs.add(refVO);
      }
    }else{
      String keyName = this.name;
      if (!modificationSite.equalsIgnoreCase("*")) keyName+="("+this.modificationSite+")";
      Double[] masses = (Double[])SmModificationVO.decodeFormula(this.formula,elements).get(0);
      MassreferenceVO monoRefVO = new MassreferenceVO(null,keyName,null,masses[0],null);
      refs.add(monoRefVO);
    }
    return refs;
  }
  
  /**
   * @param otherModifications all the modifications
   * @param elements the chemical elements to calculate the modification values
   * @return the average mass reference VOs
   * @throws SpectrummillParserException
   */
  public Vector<MassreferenceVO> getAverageMassreferences(Hashtable<String,SmModificationVO> otherModifications, Hashtable<String,SmChemicalElementVO> elements) throws SpectrummillParserException{
    Vector<MassreferenceVO> refs = new Vector<MassreferenceVO>();
    if (this.type.equalsIgnoreCase(GlobalConstants.SM_CONFIG_XML_MODIFICATION_TYPE_VALUE_CYCLIC)){
      for (String id : this.cyclicModificationIds){
        if (!otherModifications.containsKey(id)) throw new SpectrummillParserException("The cyclic modification "+this.id+"references to the modification"+id+" which does not exist");
        Vector<MassreferenceVO> cyclicRefs = otherModifications.get(id).getAverageMassreferences(otherModifications,elements);
        for (MassreferenceVO refVO : cyclicRefs) refs.add(refVO);
      }
    }else{
      String keyName = this.name;
      if (!modificationSite.equalsIgnoreCase("*")) keyName+="("+this.modificationSite+")";
      Double[] masses = (Double[])SmModificationVO.decodeFormula(this.formula,elements).get(0);
      MassreferenceVO monoRefVO = new MassreferenceVO(null,keyName,null,masses[1],null);
      refs.add(monoRefVO);
    }
    return refs;
  }
  
  /**
   * Decodes the chemical formula for the modifications and calculates the modification value
   * @param formula chemical formula
   * @param chemElements the chemical elements
   * @return the mass values ([0]: monoisotopic [1]: average)
   * @throws SpectrummillParserException
   */
  public static Vector decodeFormula(String formula, Hashtable<String,SmChemicalElementVO> chemElements) throws SpectrummillParserException{
    Vector massesAndElements = new Vector();
    Double[] masses = new Double[2];
    Vector<SmChemicalElementVO> elements = new Vector<SmChemicalElementVO>();
    double monoisotopicMass = 0;
    double averageMass = 0;
    StringTokenizer tokenizer = new StringTokenizer(formula," ");
    while (tokenizer.hasMoreTokens()){
      String[] symbolAndAmount = SmModificationVO.getAmountOfElements(tokenizer.nextToken(), formula);
      String symbol = symbolAndAmount[0];
      int amount = Integer.parseInt(symbolAndAmount[1]);

      if (!chemElements.containsKey(symbol)) throw new SpectrummillParserException("The formula "+formula+" contains a element "+symbol+" which is not defined at the elements definition");

      SmChemicalElementVO chemElement = chemElements.get(symbol);
      monoisotopicMass += chemElement.getMonoMass()*amount;
      averageMass += chemElement.getAverageMass()*amount;
      if (amount>=0){
        for (int i=0; i!=amount; i++){
          elements.add(chemElement);
        }
      }else{
        for (int i=0; i!=amount; i--){
          //TODO: Normally I have not just to make a remove operation, 
          // but if nothing is there to be removed, I have to add a negative element!
          // However, wherever this is used, a negative element is not possible!
          for (int j=0; j!=elements.size();j++){
            SmChemicalElementVO otherElem = elements.get(j);
            if (otherElem.getChemicalSymbol().equals(symbol)){
              elements.remove(j);
              break;
            }
          }
        }
      }
    }
    masses[0] = monoisotopicMass;
    masses[1] = averageMass;
    massesAndElements.add(masses);
    massesAndElements.add(elements);
    return massesAndElements;
  }
  
  private static String[] getAmountOfElements(String formulaElement, String formula) throws SpectrummillParserException{
    StringTokenizer tok2 = new StringTokenizer(formulaElement,"-");
    String[] symbolAndAmount = new String[2];
    String symbol;
    int amount;
    if (tok2.countTokens()==2){
      symbol = tok2.nextToken();
      try{
        amount = Integer.parseInt(tok2.nextToken())*-1;
      }catch (NumberFormatException nfx){
        throw new SpectrummillParserException("The formula "+formula+"could not be decoded");
      }
    }else{
      int amountPosition = 0;
      char[] formulaChars = formulaElement.toCharArray();
      for (int i=0; i!=formulaChars.length;i++){
        if (Character.isDigit(formulaChars[i])){
          amountPosition = i;
          break;
        }
      }
      if (amountPosition==0){
        symbol = formulaElement;
        amount = 1;
      }else{
        symbol = formulaElement.substring(0,amountPosition);
        try{
          amount = Integer.parseInt(formulaElement.substring(amountPosition));
        }catch (NumberFormatException nfx){
          throw new SpectrummillParserException("The formula "+formula+"could not be decoded");
        }
      }
    }
    symbolAndAmount[0] = symbol;
    symbolAndAmount[1] = String.valueOf(amount);
    return symbolAndAmount;
  }
  
  /**
   * @return type of modification
   */
  public String getType()
  {
    return type;
  }

  public String getFormula()
  {
    return formula;
  }

  public String getModificationSite()
  {
    return modificationSite;
  }
  
  
  
  
}
