package at.tugraz.genome.maspectras.quantification;

import org.apache.commons.cli.BasicParser;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.log4j.Logger;

import at.tugraz.genome.maspectras.quantification.parser.AnalyzerClusterNodeParser;
import at.tugraz.genome.maspectras.quantification.exception.AnalyzerClusterNodeException;

import at.tugraz.genome.maspectras.quantification.CgAnalyzer;

import java.util.Vector;

import at.tugraz.genome.maspectras.quantification.CgParameterSet;
import at.tugraz.genome.maspectras.quantification.CgException;

import java.io.FileOutputStream;
import java.io.BufferedOutputStream;

public class AnalyzerClusterNodeAdapter
{

  /**
   * @param args
   * @param args[0] startLine
   * @param args[1] stopLine
   * @param args[2] filepath peptide file
   * @param args[3] filepath raw file
   * @param args[4] filepath output file
   */
  private static Logger log_ = Logger.getLogger(AnalyzerClusterNodeAdapter.class);
  
  private static String version = "0_3_1";
  
  /** command line tool to start the quantification on another machine (JClusterService)*/
  public static void main(String[] args)
  {
    
    Options parameters = new Options();
    Option option=new Option("b","begin", true, "begin line of of the input file");
    option.setRequired(false);
    parameters.addOption(option);
    option=new Option("e","end", true, "end line of of the input file");
    option.setRequired(false);
    parameters.addOption(option);
    option=new Option("i","inputfile", true, "[filename] file with an header element of the found peptides and the value pairs with charge and scannumber");
    option.setRequired(true);
    parameters.addOption(option);
    option=new Option("r","rawfile", true, "[filename] file with the raw data");
    option.setRequired(true);
    parameters.addOption(option);
    option=new Option("d","headerfile", true, "[filename] file with the header data");
    option.setRequired(false);
    parameters.addOption(option);
    option=new Option("x","indexfile", true, "[filename] file with the index data");
    option.setRequired(false);
    parameters.addOption(option);
    option=new Option("m","timefile", true, "[filename] file with the retention time and the original scan number");
    option.setRequired(false);
    parameters.addOption(option);
    option=new Option("o","outputfile", true, "[filename] file with an header element with the calculated area and additional information about each peak");
    option.setRequired(true);
    parameters.addOption(option);
    option=new Option("t","filetype", true, "type of the raw file could be mzXML, mzData, chrom or RAW(only 1.3 is supported)");
    option.setRequired(true);
    parameters.addOption(option);
    option=new Option("q","quantificationtype", true, "type of the quantification method "+CgDefines.StandardValleyMethod+"=ASAPRatio standard; "+CgDefines.EnhancedValleyMethod+"=ASAPRatio Enhanced Valley");
    option.setRequired(false);
    parameters.addOption(option);
    option=new Option("v","version", true, "version of msQuantification.jar");
    option.setRequired(false);
    parameters.addOption(option);
    
    
    CommandLineParser cmdParser = new BasicParser();
    CommandLine command;
    try
    {
        command = cmdParser.parse(parameters, args);
    } 
    catch (ParseException pe)
    { 
        pe.printStackTrace();
        printUsage(parameters,pe.getMessage()); 
        return; 
    }
    
    int startLine = 0;
    int stopLine = Integer.MAX_VALUE;
    String inputFilePath;
    String rawFilePath;
    String outputFilePath;
    String fileType;
    String headerFilePath = null;
    String indexFilePath = null;
    String retentionTimeFilePath = null;
    Integer quantificationType = new Integer(CgDefines.StandardValleyMethod);    
/*    if (args.length>5){
      log_.error("Too many parameters");
      System.exit(1);
    }
    if (args.length<5) {
      log_.error("Not enough parameters");
      System.exit(1);
    }*/
    if (command.getOptionValue("v")!=null){
      System.out.println("Version: "+AnalyzerClusterNodeAdapter.version);
      System.exit(0);
    }
    try{
      if (command.getOptionValue("b")!=null) startLine = new Integer(command.getOptionValue("b"));
    }catch (NumberFormatException nfx){
      log_.error("Start Line is not Integer Format");
      System.exit(1);
    }
    try{
      if (command.getOptionValue("e")!=null) stopLine = new Integer(command.getOptionValue("e"));
    }catch (NumberFormatException nfx){
      log_.error("Stop Line is not Integer Format");
      System.exit(1);
    }
    inputFilePath = command.getOptionValue("i");
    rawFilePath = command.getOptionValue("r");
    outputFilePath = command.getOptionValue("o");
    fileType = command.getOptionValue("t");
    if (fileType.equalsIgnoreCase("chrom")){
      if (command.getOptionValue("d")!=null) headerFilePath = command.getOptionValue("d");
      else{
        log_.error("When a chromatography file is used, a header file path must be defined!");
        System.exit(1);
      }
      if (command.getOptionValue("x")!=null) indexFilePath = command.getOptionValue("x");
      else{
        log_.error("When a chromatography file is used, an index file path must be defined!");
        System.exit(1);
      }
      if (command.getOptionValue("m")!=null) retentionTimeFilePath = command.getOptionValue("m");
      else{
        log_.error("When a chromatography file is used, a retention time file path must be defined!");
        System.exit(1);
      }
    }
    if (command.getOptionValue("q")!=null){
      try{
        quantificationType = new Integer(command.getOptionValue("q"));
        if (quantificationType.intValue()!=CgDefines.StandardValleyMethod&&quantificationType.intValue()!=CgDefines.EnhancedValleyMethod){
          log_.error("The value "+quantificationType+" is not allowed for the parameter \"q\" setting it to "+CgDefines.StandardValleyMethod);
          quantificationType = CgDefines.StandardValleyMethod;
        }
      }catch(java.lang.NumberFormatException nfx){
        log_.error("The value "+command.getOptionValue("q")+" is not allowed for the parameter \"q\" setting it to "+CgDefines.StandardValleyMethod);
        nfx.printStackTrace();
      }
    }
    AnalyzerClusterNodeParser parser = new AnalyzerClusterNodeParser(inputFilePath,startLine,stopLine);
    String outputString = "";
    try{
      Analyzer analyzer = AnalyzerClusterFactory.getAnalyzer(rawFilePath, headerFilePath, indexFilePath, retentionTimeFilePath, fileType);
      parser.parse();
      Vector quanParams = parser.getCgParameters();
      for (int i=0; i!= quanParams.size();i++){
        CgParameterSet set = (CgParameterSet) quanParams.get(i);
        set.ValleyMethod = quantificationType.intValue();
        analyzer.Process(set);
        if (analyzer.getTotalArea()!=Float.NaN&&analyzer.getTotalArea()!=0){
          outputString += ">"+set.Peptide;
          outputString += "\t"+analyzer.getTotalArea()+"\n";
          CgParameterSet calculatedSet = analyzer.getM_params();
          for (int j=0; j!=calculatedSet.ProbeCount();j++){
            CgProbe aProbe = calculatedSet.Probe(j);
            if (aProbe.AreaStatus==CgAreaStatus.OK){
              outputString += aProbe.Charge+"\t"+aProbe.LowerValley+"\t"+aProbe.UpperValley+"\t"+aProbe.Area+"\t"+aProbe.Peak+"\t"+aProbe.Background+"\t"+
                aProbe.AreaError+"\n";
            }
          }
        }
      }
// write the resultFile
      BufferedOutputStream stream = new BufferedOutputStream(new FileOutputStream(outputFilePath));
      stream.write(outputString.getBytes());
      stream.close();
      stream.flush();
    } catch (AnalyzerClusterNodeException acnx){
      acnx.printStackTrace();
      System.exit(1);
    } catch (CgException cx){
      cx.printStackTrace();
      System.exit(1);
    }catch (Exception ex){
      ex.printStackTrace();
      System.exit(1);
    }
    
  }
  
    private static void printUsage(Options options, String message)
    {
        HelpFormatter formatter = new HelpFormatter();
        formatter.printHelp( "analyzerClusterNodeAdapter... calculates the peptide peak areas and returns\n\nERROR: "+message+"\n\n", options );
    }

}
