package at.tugraz.genome.maspectras.rawfilereader;

import java.io.InputStream;
import java.io.IOException;
import java.io.RandomAccessFile;

import java.util.logging.Level;
import java.util.logging.Logger;

public class MSBinaryParser {

  private int		forceversion=-1;

  private Logger	logger;

  public MSBinaryParser() {
    logger=Logger.getLogger(MSBinaryParser.class.getName());
  }

  public void setForceVersion(String version) {
    if("1.2".equals(version)) forceversion=0x2f;
    else if("1.3".equals(version)) forceversion=0x32;
    else if("1.4".equals(version)) forceversion=0x39;
    else throw new IllegalArgumentException("Invalid version "+version);
  }


  public MSBinaryData parseFile(RandomAccessFile datafile)
                      throws MSBinaryParserException {

    byte buffer[]=null;
    try {
      buffer=new byte[(int)(datafile.length())];
      datafile.seek(0);
      int i=datafile.read(buffer);
      if(i!=buffer.length) {
        throw new MSBinaryParserException("Failed to read file");
      }
    } catch (IOException ioe) {
      logger.log(Level.SEVERE, "IOException reading data from "+datafile, ioe);
      throw new MSBinaryParserException("IOException while reading from file");
    }

    String str=null;
    if((getShort(buffer, 0)!=0xa101)||
       (!"Finnigan".equals(str=getString(buffer, 2, 16)))) {
      throw new MSBinaryParserException("Wrong header signature");
    }

    int parsepos=0x22;
    if(getShort(buffer, parsepos)!=0x8) {
      throw new MSBinaryParserException("Unexpected header bytes");
    }
    parsepos+=2;

    int version=getInt(buffer, parsepos);
    if((forceversion>=0)&&(forceversion!=version)) {
      logger.info("Forcing version code 0x"+Integer.toHexString(forceversion)+
                  " instead of 0x"+Integer.toHexString(version));
      version=forceversion;
    }
    parsepos+=4;

    switch(version) {
      case 0x2f: // version 1.2???
        return(new MSBinaryDataV1d2(buffer));
      case 0x32: // version 1.3???
        return(new MSBinaryDataV1d3(buffer));
      case 0x39: // version 1.4???
        return(new MSBinaryDataV1d4(buffer));
      default:
        throw new MSBinaryParserException("Unknown version code 0x"+
                                          Integer.toHexString(version));
    }


  }





  public static int getShort(byte data[], int pos) {
    return(((data[pos+1]&0xFF)<<8)|(data[pos]&0xFF));
  }

  public static int getInt(byte data[], int pos) {
    int val=0;
    for(int xp=pos+3; xp>=pos; xp--) {
      val=(val<<8)|(data[xp]&0xFF);
    }
    return(val);
  }

  public static long getLong(byte data[], int pos) {
    long val=0;
    for(int xp=pos+7; xp>=pos; xp--) {
      val=(val<<8)|(data[xp]&0xFF);
    }
    return(val);
  }


  public static float getFloat(byte data[], int pos) {
    int val=0;
    for(int xp=pos+3; xp>=pos; xp--) {
      val=(val<<8)|(data[xp]&0xFF);
    }
    return(Float.intBitsToFloat(val));
  }

  public static double getDouble(byte data[], int pos) {
    long val=0;
    for(int xp=pos+7; xp>=pos; xp--) {
      val=(val<<8)|(data[xp]&0xFF);
    }
    return(Double.longBitsToDouble(val));
  }

  public static String getString(byte data[], int pos, int chars) {
    char strc[]=new char[chars];
    int lastnonnull=-1;
    for(int cpos=0; cpos<chars; cpos++) {
      char c=(char)((data[pos++]&0xFF)|(data[pos++]<<8));
      if(c!=0) lastnonnull=cpos;
      strc[cpos]=c;
    }
    return(new String(strc, 0, ++lastnonnull));
  }



}

