package at.tugraz.genome.maspectras.utils;

/**
 * <p>Title: Calculator </p>
 * <p>Description: Calculates some marsQM specific values</p>
 * <p>Copyright: Copyright (c) 2003</p>
 * <p>Company: TU-Graz, Institut</p>
 * @ Hartler Juergen
 * @version 1.0
 */


import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.util.Arrays;
import java.util.StringTokenizer;
import java.util.Vector;
import java.util.Collections;

import JSci.maths.statistics.TDistribution;
import at.tugraz.genome.util.FloatMatrix;

/**
 * Basic accumulation for calculation tools 
 * @author Juergen Hartler
 *
 */
public class Calculator {

  /** pattern for scientific format*/
  private final static DecimalFormat scientificFormat_ = new DecimalFormat("0.0E0"); 
  
  /**
   * Formats a double number to a definable amount of decimal places
   * @param expr the double value
   * @param decplaces amount of decimal places
   * @return the formated number
   */
  public static Double FormatNumber(Double expr, Double decplaces){
    return new Double(Math.rint(expr.doubleValue() * Math.pow(10,decplaces.doubleValue()))/Math.pow(10,decplaces.doubleValue()));
  }

  /**
   * Formats a double number to a definable amount of decimal places
   * @param expr the double value
   * @param decplaces amount of decimal places
   * @return the formated number
   */
  public static double FormatNumber(double expr, double decplaces){
    return (Math.rint(expr * Math.pow(10,decplaces))/Math.pow(10,decplaces));
  }
  
  public static String FormatNumberToString(double expr, double decplaces){
    String formattedNumber1 = (new BigDecimal((Math.rint(expr * Math.pow(10,decplaces))/Math.pow(10,decplaces)))).toPlainString();
    String formattedNumber2 = formattedNumber1;
    if (decplaces<1){
      if (formattedNumber1.indexOf(".")!=-1)
        formattedNumber2 = formattedNumber1.substring(0,formattedNumber1.indexOf("."));
    }else{
      int dotIndex = formattedNumber1.indexOf(".");
      if (dotIndex!=-1){
        formattedNumber2 = formattedNumber1.substring(0,formattedNumber1.indexOf("."));
      }else{
        dotIndex = formattedNumber1.length();
      }
      formattedNumber2+=".";
      for (int i=0;i!=decplaces;i++){
        if (i+1+dotIndex>(formattedNumber1.length()-1))
          formattedNumber2+="0";
        else
          formattedNumber2+=formattedNumber1.substring(i+1+dotIndex,i+2+dotIndex); 
      }
    }
    return formattedNumber2;
  }
  
  public static int formatNumberToInteger(double expr){
    return Integer.parseInt(FormatNumberToString(expr,0));
  }


  /**
   * Rounds the decimal places 
   * @param targetDBL the double value
   * @param decimalPlace amount of decimal places
   * @return the rounded number
   */
  public static double roundDBL(double targetDBL,int decimalPlace){
    return roundDBL(targetDBL,decimalPlace,BigDecimal.ROUND_UP);
  }

  public static double roundDBL(double targetDBL,int decimalPlace,int roundMode){
    BigDecimal bd = new BigDecimal(targetDBL);
    bd = bd.setScale(decimalPlace,roundMode);
    return (bd.doubleValue());
}

  
  /**
   * Rounds the decimal places 
   * @param targetDBL the double value
   * @param decimalPlace amount of decimal places
   * @return the rounded number
   */
  public static float roundFloat(float targetFloat,int decimalPlace){
    return Calculator.roundFloat(targetFloat,decimalPlace,BigDecimal.ROUND_HALF_UP);
  }

  
  /**
   * Rounds the decimal places 
   * @param targetDBL the double value
   * @param decimalPlace amount of decimal places
   * @return the rounded number
   */
  public static float roundFloat(float targetFloat,int decimalPlace, int roundingMode){
    if (Float.isInfinite(targetFloat)||Float.isNaN(targetFloat)){
      return targetFloat;
    }else{  
      BigDecimal bd = new BigDecimal(targetFloat);
      bd = bd.setScale(decimalPlace,roundingMode);
      return (bd.floatValue());
    }  
  }

  
  /**
   * this method calculates the index of a peptide in a protein sequence
   * @param proteinSequence the whole sequence of the protein
   * @param pepSequence the sequence of the peptide
   * @return the indices where this peptide has been found
   */
  public static Vector calculateIndex(String proteinSequence,String pepSequence){
    
    Vector foundIndices = new Vector();
//    if (index==-1){
      if (proteinSequence!=null&&proteinSequence.length()>0&&pepSequence!=null&&pepSequence.length()>0){
        if (pepSequence.indexOf("X")!=-1||proteinSequence.indexOf("X")!=-1){
          StringTokenizer proteinTokenizer = new StringTokenizer(proteinSequence,"X");
          StringTokenizer pepTokenizer = new StringTokenizer(pepSequence,"X");
          String[] protTokens = new String[proteinTokenizer.countTokens()];
          String[] pepTokens = new String[pepTokenizer.countTokens()];
          int count=0;
          while (proteinTokenizer.hasMoreElements()){
            protTokens[count]=proteinTokenizer.nextToken();
            count++;
          }
          Vector proteinXIndices = new Vector();
          Vector peptideXIndices = new Vector();
          String subProteinString = new String(proteinSequence);
          while (subProteinString.indexOf("X")!=-1){
            int add=0;
            if (proteinXIndices.size()>0) add=((Integer)proteinXIndices.get(proteinXIndices.size()-1)).intValue();
            proteinXIndices.add(new Integer(subProteinString.indexOf("X")+add+1));
            subProteinString = subProteinString.substring(subProteinString.indexOf("X")+1);
          }
          String subPeptideString = new String(pepSequence);
          while (subPeptideString.indexOf("X")!=-1){
            int add=0;
            if (peptideXIndices.size()>0) add=((Integer)peptideXIndices.get(peptideXIndices.size()-1)).intValue();
            peptideXIndices.add(new Integer(subPeptideString.indexOf("X")+add+1));
            subPeptideString = subPeptideString.substring(subPeptideString.indexOf("X")+1);
          }
          count=0;
          while (pepTokenizer.hasMoreElements()){
            pepTokens[count]=pepTokenizer.nextToken();
            count++;
          }
          int xCount = proteinSequence.indexOf(protTokens[0]);
          for (int k=0; k!=protTokens.length;k++){           
            int[] numberPlusLastX = Calculator.calculateNumberOfFollowingXs(proteinXIndices, xCount);
//            int numberOfFollowingXs = numberPlusLastX[0];
            int lastXCount = numberPlusLastX[1]/*+numberPlusLastX[0]*/;
//first look if a sequence with xs is within the proteinfragment
            String subProteinFragmentString = new String(protTokens[k]);
            int substringCount = 0;
            while (subProteinFragmentString.indexOf(pepTokens[0])!=-1){             
              int addToSubString = subProteinFragmentString.indexOf(pepTokens[0])+pepTokens[0].length();//pepSequence.length();
              if (subProteinFragmentString.length()>subProteinFragmentString.indexOf(pepTokens[0])+pepTokens[0].length()+1){
                String subProteinFragmentDividedString = subProteinFragmentString.substring(subProteinFragmentString.indexOf(pepTokens[0])+pepTokens[0].length()+1);
                int pepCount = 1;
                int peptideXCount = pepSequence.indexOf(pepTokens[0]);
                while (((pepCount)<pepTokens.length)&&subProteinFragmentDividedString.startsWith(pepTokens[pepCount])){
                  int[] numberPlusLastX2 = Calculator.calculateNumberOfFollowingXs(peptideXIndices, (peptideXCount));
                  int peptideLastXCount = numberPlusLastX2[1];
                  subProteinFragmentDividedString = subProteinFragmentDividedString.substring(pepTokens[pepCount].length()+numberPlusLastX2[0]);
                  pepCount++;
                  if (pepCount==pepTokens.length){
                    int subIndex=substringCount+subProteinFragmentString.indexOf(pepTokens[0]);
                    foundIndices.add(new Integer(Calculator.calculateTotalIndex(protTokens,k,xCount,subIndex-pepSequence.indexOf(pepTokens[0]))));
                  }
                  peptideXCount = peptideLastXCount;
                }
                if (pepTokens.length==1){
                  foundIndices.add(new Integer(Calculator.calculateTotalIndex(protTokens,k,xCount,subProteinFragmentString.indexOf(pepTokens[0])-pepSequence.indexOf(pepTokens[0]))));
                }
                substringCount = substringCount+addToSubString;
//                if (subProteinFragmentString.length()>subProteinFragmentString.indexOf(pepTokens[0])+pepSequence.length()){
//                  subProteinFragmentString = subProteinFragmentString.substring(subProteinFragmentString.indexOf(pepTokens[0])+pepSequence.length());
//                }else{
                subProteinFragmentString = subProteinFragmentString.substring(subProteinFragmentString.indexOf(pepTokens[0])+pepTokens[0].length());
//                }
              }else{
                if (pepTokens.length==1){
                  foundIndices.add(new Integer(Calculator.calculateTotalIndex(protTokens,k,xCount,subProteinFragmentString.indexOf(pepTokens[0]))));
                }
                break;
              }
            }
           
           
//          shuffle single pepSubStrings over the protSequence
            if (xCount<proteinXIndices.size()){
//              Vector<Integer> possibleEndPositions = new Vector<Integer>();
              char[] pepChars = pepSequence.toCharArray();
              //look if the following sequence starting with Xs equals the sequence
              for (int l=-numberPlusLastX[0];l!=pepSequence.length()-1;l++){
                String compareStringEnd ="";
                if (protTokens[k].length()>l&&l>-1){
                  for (int m=0;m!=l+1;m++){
                    if (pepChars[m]=='X'){
//                    Substitution of X                            
                      compareStringEnd = compareStringEnd+String.valueOf(protTokens[k].length()-l+m-1);
                    }else compareStringEnd = compareStringEnd+String.valueOf(pepChars[m]);
                  }
                }
                if ((protTokens[k].length()>=compareStringEnd.length())&&((compareStringEnd.length()>0&&protTokens[k].substring(protTokens[k].length()-compareStringEnd.length(),protTokens[k].length()).equalsIgnoreCase(compareStringEnd))||(l<0))){
                  String protString = "";
                  if (l>-1){
                    protString = new String(compareStringEnd);
                  }  
                  int additionalAddedStringNumber = 0;
                  int lastXCount2 = (new Integer(xCount)).intValue();
                  while (((k+additionalAddedStringNumber)<protTokens.length)&&(protString.length()<=pepSequence.length())){
                    int[] numberPlusLastX2 = Calculator.calculateNumberOfFollowingXs(proteinXIndices, lastXCount2);
                    int numberOfFollowingXs = numberPlusLastX2[0];
                    if (additionalAddedStringNumber==0&&l<0){
                      numberOfFollowingXs = numberOfFollowingXs+l+1;
                    }
                    lastXCount2 = numberPlusLastX2[1];
                    for (int m=0; m!=numberOfFollowingXs;m++){
                      if (pepChars.length>protString.length()){
                        protString = protString+pepChars[protString.length()];
                      }else{
                        protString = protString+"X";
                      }
                    }
                    additionalAddedStringNumber++;
                    if (k+additionalAddedStringNumber<protTokens.length){
                      protString = protString+protTokens[k+additionalAddedStringNumber];
                    }  
                  }
//now removing the Xs from the pepChars                        
                  if (protString.length()>=pepChars.length){
                    String compareString = "";
                    protString=protString.substring(0,pepChars.length);
                    for (int m=0;m!=pepChars.length;m++){
                      if (pepChars[m]=='X'){
//                      Substitution of X                            
                        compareString = compareString+String.valueOf(protString.toCharArray()[m]);
                      }else compareString = compareString+String.valueOf(pepChars[m]);
                    }
                    if (protString.equalsIgnoreCase(compareString)){
                      foundIndices.add(new Integer(Calculator.calculateTotalIndex(protTokens,k,xCount,protTokens[k].length()-l-1)));
                    }
                  }
                }
              }
              
            }
            xCount = lastXCount;
          }
        }else{
          String subProteinString = new String(proteinSequence);         
          while (subProteinString.indexOf(pepSequence)!=-1){
            int add=0;
            if (foundIndices.size()>0) add=((Integer)foundIndices.get(foundIndices.size()-1)).intValue();
            foundIndices.add(new Integer(subProteinString.indexOf(pepSequence)+add+(foundIndices.size()*pepSequence.length())));
            subProteinString = subProteinString.substring(subProteinString.indexOf(pepSequence)+pepSequence.length());
          }
        }
      }else{
        foundIndices.add(new Integer(-1));
      }
//    }
//    int index = -1;//proteinSequence.indexOf(pepSequence);
    if (foundIndices.size()<1){
      foundIndices.add(new Integer(-1));
    }
    return foundIndices;
  }
  
  /**
   * calculates the total index in the protein and not the relative one
   * @param protTokens the segmentated protein tokens
   * @param fragmentNumber the current fragment number
   * @param xsBefore amount of X values before
   * @param indexInFragment relative index in the fragment
   * @return
   */
  private static int calculateTotalIndex(String[] protTokens, int fragmentNumber, int xsBefore, int indexInFragment){
    int index = 0;
    for (int i=0; i!=(fragmentNumber);i++){
      index = index+protTokens[i].length();
    }
    index = index+xsBefore+indexInFragment;
    if (index<-1){
      index=-1;
    }
    return index;
  }
  
  /**
   * calculates the index of the next X
   * @param proteinXIndices the indices of the Xs
   * @param xCount the actual count of the X
   * @return the index of the following X[0] the last count of the Xs
   */
  private static int[] calculateNumberOfFollowingXs(Vector proteinXIndices, int xCount){
    int numberOfFollowingXs = 0;
    int lastXCount = (new Integer(xCount)).intValue();
    if (lastXCount<proteinXIndices.size()) numberOfFollowingXs++;
    while (((lastXCount+1)<proteinXIndices.size())&&((((Integer)proteinXIndices.get(lastXCount)).intValue()+1)==((Integer)proteinXIndices.get(lastXCount+1)).intValue())){
      lastXCount++;
      numberOfFollowingXs++;
    }
    if (lastXCount<proteinXIndices.size()) lastXCount++;
    int[] result = new int[2];
    result[0]=numberOfFollowingXs;
    result[1]=lastXCount;
    return result;
  }
  
  /**
   * calculates the variance 
   * @param x_in the values
   * @return the variance
   */
  public static float variance(float[] x_in) {
    float varia = 0f;
    float mu = 0f;
    int samp_size = 0;
    float val;

    mu = mean(x_in);

    if (x_in.length > 1) {

        for (int cnt = 0; cnt < x_in.length; cnt++) {
            val = x_in[cnt];
            if (!Float.isNaN(val)) {
                samp_size++;
                varia += Math.pow((val - mu), 2);
            }
        }

        varia = varia / (samp_size - 1);
        return varia;
    } else {
        return Float.NaN;
    }
    
  }
  
  public static double variance(double[] x_in) {
    double varia = 0f;
    double mu = 0f;
    int samp_size = 0;
    double val;

    mu = mean(x_in);

    if (x_in.length > 1) {

        for (int cnt = 0; cnt < x_in.length; cnt++) {
            val = x_in[cnt];
            if (!Double.isNaN(val)) {
                samp_size++;
                varia += Math.pow((val - mu), 2);
            }
        }

        varia = varia / (samp_size - 1);
        return varia;
    } else {
        return Double.NaN;
    }
    
  }
  
  /**
   * the mean of a list of values
   * @param x the values
   * @return the mean value
   */
  public static float mean(float[] x) {

    int x_len;
    float x_mean = 0.0f;
    float val;
    x_len = 0;

    for (int i = 0; i < x.length; i++) {
        val = x[i];

        if (!Float.isNaN(val)) {
            x_len++;
            x_mean = x_mean + val;
        }
    }
    if (x_len>0)
      x_mean = x_mean / x_len;
    else
      x_mean = Float.NaN;
    return x_mean;
}
  
  public static double mean(double[] x) {

    int x_len;
    double x_mean = 0.0f;
    double val;
    x_len = 0;

    for (int i = 0; i < x.length; i++) {
        val = x[i];

        if (!Double.isNaN(val)) {
            x_len++;
            x_mean = x_mean + val;
        }
    }
    if (x_len>0)
      x_mean = x_mean / x_len;
    else
      x_mean = Double.NaN;
    return x_mean;
}
  
  public static float mean(Vector<Double> values){
    float[] x = new float[values.size()];
    for (int i=0; i!=values.size();i++)
      x[i] = values.get(i).floatValue();
    return Calculator.mean(x);  
  }

  public static double stddeviation(Vector<Double> values) {
    double[] x_in = new double[values.size()];
    for (int i=0; i!=values.size();i++)
      x_in[i] = values.get(i).floatValue();
    return Math.sqrt(variance(x_in));
 }
  
  /**
   * @param x_in the values
   * @return the standard deviation
   */
  public static float stddeviation(float[] x_in) {
    return (float) Math.sqrt(variance(x_in));
}
  
  /**
   * calculates the parameters of a regression line
   * @param x_in the x values
   * @param y_in the y values
   * @return the regression parameters (a,b,r)
   */
  public static double[] calcLinRegression(double[] x_in, double[] y_in) {

    double[] data_out = new double[3];

    int N = 0;
    int num_points_in = 0;
    double x;
    double y;
    double sumx = 0.0f;
    // sum  x[i],      i = 1..n
    double sumy = 0.0f;
    // sum  y[i],      i = 1..n
    double sumx2 = 0.0f;
    // sum  x[i]*x[i], i = 1..n
    double sumy2 = 0.0f;
    // sum  y[i]*y[i], i = 1..n
    double sumxy = 0.0f;
    // sum  x[i]*y[i], i = 1..n
    x = 0f;
    y = 0f;

    // get num of points:
    num_points_in = x_in.length;

    // read data and compute statistics
    for (int i = 0; i < num_points_in; i++) {

      x = x_in[i];
      y = y_in[i];

      //just count reliable values!!
      //ignore NaNs -> calculate correct num of points.
      if (!Double.isNaN(x) && !Double.isNaN(y)) {
        sumx += x;
        sumy += y;
        sumx2 += x * x;
        sumy2 += y * y;
        sumxy += x * y;
        N++;
      }
    }

    // compute coefficients
    double a = (N * sumxy - sumx * sumy) / (N * sumx2 - sumx * sumx);
    double b = (sumy - a * sumx) / N;
    double r = (N * sumxy - sumx * sumy) /
        (double) Math.sqrt( (N * sumx2 - sumx * sumx) *
                           (N * sumy2 - sumy * sumy));

    data_out[0] = a;
    //slope
    data_out[1] = b;
    //intercept
    data_out[2] = r;
    //coefficient of determination

    return data_out;
  }
 
  /**
   * calculates the median
   * @param values values
   * @return median
   */
  public static Float median(Vector<Float> values)
  {
    Collections.sort(values);
    Float median;
    if (values.size() > 0) {
      if (values.size() % 2 == 0) {
        median = new Float(((values.get((values.size() / 2) - 1))
            .floatValue() + (values.get(values.size() / 2)).floatValue()) / 2);
      } else {
        median = new Float(values.get(((values.size() + 1) / 2) - 1));
      }
    } else {
      median = new Float(0);
    }
    return median;
  }
  
  /**
   * calculates the median
   * @param values values
   * @return median
   */
  public static Float median(float[] values)
  {
    Arrays.sort(values);
    Float median;
    if (values.length > 0) {
      if (values.length % 2 == 0) {
        median = new Float((values[(values.length/2)-1]+values[values.length/2])/2);
      } else {
        median = new Float(values[((values.length + 1)/2)-1]);
      }
    } else {
      median = new Float(0);
    }
    return median;
  }
  
  public static Float[] medianPlusUpperLowerValues(Vector<Float> values)
  {
    Collections.sort(values);
    Float[] medians = new Float[3];
    Float median;
    Float lowerSectionMedian;
    Float upperSectionMedian;
    if (values.size() % 2 == 0) {
      median = new Float(((values.get((values.size() / 2) - 1)).floatValue() + (values.get(values.size() / 2)).floatValue()) / 2);
    } else {
      median = new Float(values.get(((values.size() + 1) / 2) - 1));
    }
    int position = values.size() / 4;
    lowerSectionMedian = new Float(((values.get((position))).floatValue() + (values.get(position+1)).floatValue()) / 2);
    position = (3*values.size())/4;
    upperSectionMedian = new Float(((values.get((position-1))).floatValue() + (values.get(position)).floatValue()) / 2);
    medians[0] = median;
    medians[1] = lowerSectionMedian;
    medians[2] = upperSectionMedian;
    return medians;    
  }
  
  
  /**
   * Calculates the median plus the median of the upper and lower section (upper and lower section median is not the direct median
   * since in the comparison of intensity values big ratios are not so easily detectable)
   * @param values the values
   * @return the median plus the median of the upper and lower section
   */
  public static Float[] medianPlusWeighedUpperLowerValues(Vector<Float> originalValues)
  {
    Vector<Float> values = new Vector<Float>(originalValues);
    Collections.sort(values);
    Float[] medians = new Float[3];
    Float median;
    Float lowerSectionMedian;
    Float upperSectionMedian;
    if (values.size() > 0) {
      if (values.size() % 2 == 0) {
        median = new Float(((values.get((values.size() / 2) - 1))
            .floatValue() + (values.get(values.size() / 2)).floatValue()) / 2);
      } else {
        median = new Float(values.get(((values.size() + 1) / 2) - 1));
      }
      if (median<1){
        int position = values.size() / 4;
        if (position==0){
          if (values.size()>1){
            lowerSectionMedian = new Float(((values.get((position))).floatValue()
                + (values.get(position+1)).floatValue()) / 2);
          }else{
            lowerSectionMedian = new Float(values.get(position));
          }
        }else{
          lowerSectionMedian = new Float(((values.get((position-1))).floatValue()
              + (values.get(position)).floatValue()) / 2);          
        }
        position = (2*values.size())/3;
        if (values.size()>1){
          upperSectionMedian = new Float(((values.get((position-1))).floatValue()
              + (values.get(position)).floatValue()) / 2);
        }else{
          upperSectionMedian = new Float(values.get(position));
        }        
      }else{
        int position = values.size() / 3;
        if (position==0){
          if (values.size()>1){
            lowerSectionMedian = new Float(((values.get((position))).floatValue()
                + (values.get(position+1)).floatValue()) / 2);
          }else{
            lowerSectionMedian = new Float(values.get(position));
          }
        }else{
          lowerSectionMedian = new Float(((values.get((position-1))).floatValue()
              + (values.get(position)).floatValue()) / 2);          
        }
        position = (3*values.size())/4;
        if (values.size()>1){
          upperSectionMedian = new Float(((values.get((position-1))).floatValue()
              + (values.get(position)).floatValue()) / 2);
        }else{
          upperSectionMedian = new Float(values.get(position));
        }        
        
      }
    } else {
      median = new Float(0);
      lowerSectionMedian = new Float(0);
      upperSectionMedian = new Float(0);
      
    }
    medians[0] = median;
    medians[1] = lowerSectionMedian;
    medians[2] = upperSectionMedian;
    return medians;
  }
  
  public static double calculateNOverK(long n, long k){
    long nMinusK = n-k;
    if (nMinusK>k){
      return Calculator.calculateFaculty(n, nMinusK)/(Calculator.calculateFaculty(k, 0));     
    }else{
      return Calculator.calculateFaculty(n, k)/(Calculator.calculateFaculty(nMinusK, 0));     
    }
  }

  public static double calculateFaculty(long n, long stopValue){
    if (n>stopValue){
      return n*Calculator.calculateFaculty(n-1, stopValue);
    }else{
      return 1;
    }
  }
  
  /*** this method returns the paramaters of the ellipse by the knowledge of 4 points the 
   * and the assumption that the ellipse that the main coordinate axis are x and y
   * the points have to be equidistantly from the middle
   * @param x1 x-coordinate of the point with the y axis minimum known
   * @param y1 y-coordinate of the point with the y axis minimum known
   * @param x2 x-coordinate of the point with the y axis maximum known
   * @param y2 y-coordinate of the point with the y axis maximum known
   * @param x3 x-coordinate of the point with the x axis minimum known
   * @param y3 y-coordinate of the point with the x axis minimum known
   * @param x4 x-coordinate of the point with the x axis maximum known
   * @param y4 y-coordinate of the point with the x axis maximum known
   * @return
   */
  public static float[] calculateEllipseFrom4Points(float x1,float y1, float x2, float y2, float x3, float y3, float x4, float y4){
//    System.out.println("P1: "+x1+";"+y1);
//    System.out.println("P2: "+x2+";"+y2);
//    System.out.println("P3: "+x3+";"+y3);
//    System.out.println("P4: "+x4+";"+y4);
    // first the center positions of x and y are calculated, then the coordinates are centered around this region
    // the multiplication factor is then used to make comparable coordinates and to prevent rounding mistakes
    float x0 = (x4+x3)/2;
    float y0 = (y2+y1)/2;
    float averageDistanceX = ((x4-x0)+(x0-x3))/2;
    float averageDistanceY = ((y2-y0)+(y0-y1))/2;
//    System.out.println("averageDistanceX: "+averageDistanceX);
//    System.out.println("averageDistanceY: "+averageDistanceY);
    float multFactorX = Calculator.getMultiplicationFactor(averageDistanceX);
    float multFactorY = Calculator.getMultiplicationFactor(averageDistanceY);
//    System.out.println("multFactorX: "+multFactorX);
//    System.out.println("multFactorY: "+multFactorY);
    float x1Corr = (x1-x0)*multFactorX;
    float x2Corr = (x2-x0)*multFactorX;
    float x3Corr = (x3-x0)*multFactorX;
    float x4Corr = (x4-x0)*multFactorX;
    float y1Corr = (y1-y0)*multFactorY;
    float y2Corr = (y2-y0)*multFactorY;
    float y3Corr = (y3-y0)*multFactorY;
    float y4Corr = (y4-y0)*multFactorY;
    float x0Corr = (x0-x0)*multFactorX;
    float y0Corr = (y0-y0)*multFactorY;
//    System.out.println("P1: "+x1Corr+";"+y1Corr);
//    System.out.println("P2: "+x2Corr+";"+y2Corr);
//    System.out.println("P3: "+x3Corr+";"+y3Corr);
//    System.out.println("P4: "+x4Corr+";"+y4Corr);
    // this calculates the parameters of the four equations A*x2+B*y2+C*x+D*y = -1
    // in fact a 4 dimensional matrix has to be solved
    FloatMatrix  params = Calculator.getEllipseParametersFrom4Points(x1Corr,y1Corr,x2Corr,y2Corr,x3Corr,y3Corr,x4Corr,y4Corr);
////    FloatMatrix  params = Calculator.getEllipseParametersFrom4Points(x1,y1,x2,y2,x3,y3,x4,y4);
//    System.out.println("A: "+params.A[0][0]+"; B:"+params.A[1][0]+"; C:"+params.A[2][0]+"; D:"+params.A[3][0]);
    
    // with the aid of the parameters and a binomial equation the ellipse dimensions a and b can be now calculated and
    // they are afterwards transformed back to the original coordinate system
    float bHalf = params.A[2][0]/(2*params.A[0][0]);
    float q = ((float)Math.pow(y0Corr, 2)*params.A[1][0]+y0Corr*params.A[3][0]+1)/params.A[0][0];
////    float q = ((float)Math.pow(y0, 2)*params.A[1][0]+y0*params.A[3][0]+1)/params.A[0][0];
    float xLeftOuter = -bHalf-(float)Math.sqrt((Math.pow(bHalf, 2)-q));
    xLeftOuter = xLeftOuter/multFactorX+x0;
    bHalf = params.A[3][0]/(2*params.A[1][0]);
    q = ((float)Math.pow(x0Corr, 2)*params.A[0][0]+x0Corr*params.A[2][0]+1)/params.A[1][0];
////    q = ((float)Math.pow(x0, 2)*params.A[0][0]+x0*params.A[2][0]+1)/params.A[1][0];
//    System.out.println((float)Math.pow(x0Corr, 2)*params.A[0][0]+x0Corr*params.A[2][0]);
//    System.out.println("bHalf: "+bHalf+" ; "+q);
    float yLowerOuter = -bHalf-(float)Math.sqrt((Math.pow(bHalf, 2)-q));
    yLowerOuter = yLowerOuter/multFactorY+y0;
    float xMatrixDimension = (x0-xLeftOuter);
    float yMatrixDimension = (y0-yLowerOuter);
    float[] results = new float[4];
    results[0] = x0;
    results[1] = y0;
    results[2] = xMatrixDimension;
    results[3] = yMatrixDimension;
    return results;
  }
  
  private static float getMultiplicationFactor(float value){
    float multiplicationFactor = 1;
    while ((value*multiplicationFactor)>10||(value*multiplicationFactor)<1){
      if (value*multiplicationFactor>10)
        multiplicationFactor = multiplicationFactor/10;
      if (value*multiplicationFactor<1)
        multiplicationFactor = multiplicationFactor*10;
      
    }
    return multiplicationFactor;
  }

  /** solves 4D matrix of the type A*x2+B*y2+C*x+D*y = -1*/
  private static FloatMatrix getEllipseParametersFrom4Points(float x1,float y1, float x2, float y2, float x3, float y3, float x4, float y4){
    float[][] matrix = new float[4][4];
    matrix[0][0] = (float)Math.pow(x1, 2);
    matrix[0][1] = (float)Math.pow(y1, 2);
    matrix[0][2] = x1;
    matrix[0][3] = y1;
    matrix[1][0] = (float)Math.pow(x2, 2);
    matrix[1][1] = (float)Math.pow(y2, 2);
    matrix[1][2] = x2;
    matrix[1][3] = y2;
    matrix[2][0] = (float)Math.pow(x3, 2);
    matrix[2][1] = (float)Math.pow(y3, 2);
    matrix[2][2] = x3;
    matrix[2][3] = y3;
    matrix[3][0] = (float)Math.pow(x4, 2);
    matrix[3][1] = (float)Math.pow(y4, 2);
    matrix[3][2] = x4;
    matrix[3][3] = y4;
    float[] rightHandParams = new float[4];
    for (int i=0;i!=rightHandParams.length;i++){
      rightHandParams[i] = -1;
    }
    FloatMatrix floatMatrix = new FloatMatrix(matrix);
    return floatMatrix.solve(new FloatMatrix(rightHandParams,4));
  }
  
  public static Float[] calculateEllipseXBorderValues(float y, float x0, float y0, float a, float b){
    Float[] borders = new Float[2];
    float plusMinusSqrt = (a/b)*(float)Math.sqrt(Math.pow(b, 2)-Math.pow((y-y0), 2));
    borders[0] = x0-plusMinusSqrt;
    borders[1] = x0+plusMinusSqrt;
    return borders;
  }
  
  public static float calculatePValueForCorrelation (float correlationCoeff, int nrOfValues){
    float pValue = 1f;
    if (nrOfValues>2){
      int degreesOfFreedom = nrOfValues-2;
      double tValue = (correlationCoeff*Math.sqrt(degreesOfFreedom))/Math.sqrt(1-Math.pow(correlationCoeff,2));
      TDistribution dist = new TDistribution(degreesOfFreedom);
      pValue = (float)((1-dist.cumulative(tValue))*2);
    }
    return pValue;
  }
  
  public static double calculateRatioStdevErrorPropagated(Vector<Double> numerators, Vector<Double> denominators, boolean valuesMustBeBigger0, boolean correlated){
    Vector<Double> nonZeroNumerators = new Vector<Double>(numerators);
    Vector<Double> nonZeroDenominators = new Vector<Double>(denominators);
    if (valuesMustBeBigger0){
      nonZeroNumerators = new Vector<Double>();
      nonZeroDenominators = new Vector<Double>();
      for (int i=0;i!=nonZeroNumerators.size();i++){
        if (numerators.get(i)>0&&denominators.get(i)>0){
          nonZeroNumerators.add(numerators.get(i));
          nonZeroDenominators.add(denominators.get(i));
        }
      }
    }
    double[] numArray = new double[nonZeroNumerators.size()];
    double[] denArray = new double[nonZeroNumerators.size()];
    for (int i=0;i!=nonZeroNumerators.size();i++){
      numArray[i] = nonZeroNumerators.get(i);
      denArray[i] = nonZeroDenominators.get(i);
    }
    return calculateRatioStdevErrorPropagated(numArray,denArray,correlated);
  }
  
  /**
   * This method calculates the error propagated standard deviation of a ratio ratio=numerator/denominator, z=x/y
   * The law of error propagation has the following formula for the standard deviation, whereas s=standard deviation,
   * sz=standard deviation of ratio, sx=standard deviation of x, sy=standard deviation of y, dz/dx = partial derivative for the numerator, 
   * dz/dy=partial derivative of the function for the denominator, 2 (at the end)=square, rxy=regression coefficient
   * sz2 = (dz/dx)2*sx2 + (dz/dy)2sy2 + 2*dz/dx*dz/dy*sx*sy*rxy
   * whereas the term 2*dz/dx*dz/dy*rxy is just necessary if the values are correlated to one another
   * in our case z=x/y, so dz/dx = 1/y , so dz/dy = -(x/y2)
   * sz2 = sx2/y2 + (x2*sy2)/y4 - (2*x*rxy*sx*sy)/y3
   * 
   * @param numerators the numerators (x)
   * @param denominators the numerators (y)
   * @param correlated are the values correlated
   * @return
   */
  public static double calculateRatioStdevErrorPropagated(double[] numerators, double[] denominators, boolean correlated){
    if (numerators.length>1){
      double sx2 = variance(numerators);
      double sy2 = variance(denominators);
      double meanX = mean(numerators);
      double meanY = mean (denominators);
      double rxy = 0;
      if (correlated){
        rxy = calcLinRegression(numerators,denominators)[2];
      }
      return calculateRatioStdevErrorPropagated(meanX, sx2, meanY, sy2, rxy);
    }else
     return Double.NaN;
  }
  
  public static double calculateRatioStdevErrorPropagated(double valueNumerator, double varianceNumerator, 
      double valueDenominator, double varianceDenominator, double correlationCoefficent){
    //     sz2 =          sx2/y2        +                 (x2*sy2)/y4                
    double sz2 = varianceNumerator/Math.pow(valueDenominator, 2) + (Math.pow(valueNumerator, 2)*varianceDenominator)/Math.pow(valueDenominator, 4);
    //   -          (2*x*rxy*sx*sy)/y3
    sz2 -= (2*valueNumerator*correlationCoefficent*Math.sqrt(varianceNumerator)*Math.sqrt(varianceDenominator))/Math.pow(valueDenominator, 3) ;
    return Math.sqrt(sz2);
  }
  
  public static double calculateSumStdevErrorPropagated(Vector<Double> stdevs){
    double[] stdevArray = new double[stdevs.size()];
    for (int i=0; i!=stdevs.size();i++)stdevArray[i] = stdevs.get(i);
    return calculateSumStdevErrorPropagated(stdevArray);
  }
  
  public static double calculateSumStdevErrorPropagated(double[] stdevs){
    double sumVariance = 0;
    for (int i=0; i!=stdevs.length; i++){
      if (!Double.isNaN(stdevs[i]) && !Double.isInfinite(stdevs[i]))
        sumVariance+=Math.pow(stdevs[i], 2);
    }
    return Math.sqrt(sumVariance);
  }
  
  public static String scientificFormat(double value){
    return scientificFormat_.format(value);
  }
}