package at.tugraz.genome.maspectras.utils;

import java.util.Vector;

/**
 * Generic encoder to produce an unlimited amount of colors
 * @author Juergen Hartler
 *
 */
public class ColorEncoder implements java.io.Serializable

{
  /** the amount of channels corresponds to the amount of comparable features*/
  private Vector channels;
  /** comparison between the channels */
  private Vector comparedChannels;
  /** the numbers of the channels to be compared */
  private Vector comparedChannelsNumbers;
  /** the minimum rgb value */
  int rgbMinValue = 30;
  /** the maximum rgb value */
  int rgbMaxValue = 220;
  /** the range covered by one color */
  int sectionStep = rgbMaxValue - rgbMinValue;
  /** the whole range due to the mixture of several colors */
  int wholeScale = (sectionStep) * 7;

  /**
   * @param numberOfComparables number of searches to be compared
   */
  public ColorEncoder(int numberOfComparables)
  {
    channels = new Vector();
    comparedChannels = new Vector();
    comparedChannelsNumbers = new Vector();
    this.calculateChannels(numberOfComparables);
  }

  /**
   * @return direct encodings for the searches
   */
  public Vector getChannels()
  {
    return channels;
  }
  /**
   * @param channels direct encodings for the searches
   */
  public void setChannels(Vector channels)
  {
    this.channels = channels;
  }
  /**
   * @return the colors for the encoding between the channels
   */
  public Vector getComparedChannels()
  {
    return comparedChannels;
  }
  /**
   * @param comparedChannels  the colors for the encoding between the channels
   */
  public void setComparedChannels(Vector comparedChannels)
  {
    this.comparedChannels = comparedChannels;
  }
  /**
   * calculates the channels to compare
   */
  public void calculateAllComparedChannels()
  {
    Vector channelNumbers = new Vector();
    for (int i = 0; i != this.channels.size(); i++) {
      channelNumbers.add(new Integer(i + 1));
    }
    this.calculateIntermediateChannels(channelNumbers, this.channels);
  }

  /** calculates the colors for the direct channels
   * @param numberOfComparables number of channels
   */
  private void calculateChannels(int numberOfComparables)
  {
    int step = wholeScale / (numberOfComparables + 1);
    int[] valuesOnWholeScale = new int[numberOfComparables];
    for (int i = 0; i != (numberOfComparables); i++) {
      valuesOnWholeScale[i] = step * (i + 1);
      if ((numberOfComparables + 1) % 7 == 0) {
        valuesOnWholeScale[i] = valuesOnWholeScale[i]
            + (step / (((numberOfComparables + 1) / 7) + 1));
      }
    }
    this.calculateRGB(valuesOnWholeScale);
  }
  /**
   * calculates the possible combinations of the channels and their color encodings
   * @param channelNumbers the numbers for the channels
   * @param rgbs the rgb values for the channels
   */
  private void calculateIntermediateChannels(Vector channelNumbers, Vector rgbs)
  {
    if (rgbs.size() > 2) {
      for (int i = rgbs.size() - 1; i != -1; i--) {
        Vector newRgbs = new Vector();
        Vector newChannelNumbers = new Vector();
        for (int j = 0; j != rgbs.size(); j++) {
          if (j != i) {
            newRgbs.add(rgbs.get(j));
            newChannelNumbers.add(channelNumbers.get(j));
          }
        }
        this.calculateIntermediateChannels(newChannelNumbers, newRgbs);
      }
    }
    int[] rgb = { 0, 0, 0 };
    for (int i = 0; i != rgbs.size(); i++) {
      int[] oneRgb = (int[]) rgbs.get(i);
      rgb[0] = rgb[0] + oneRgb[0] / rgbs.size();
      rgb[1] = rgb[1] + oneRgb[1] / rgbs.size();
      rgb[2] = rgb[2] + oneRgb[2] / rgbs.size();
    }
    boolean exists = false;
    for (int i = 0; i != comparedChannels.size(); i++) {
      int[] alreadyCompared = (int[]) comparedChannels.get(i);
      if (alreadyCompared[0] == rgb[0] && alreadyCompared[1] == rgb[1]
          && alreadyCompared[2] == rgb[2])
        exists = true;
    }
    if (!exists) {
      comparedChannels.add(rgb);
      comparedChannelsNumbers.add(channelNumbers);
    }
    // for (int i=0; i!=)
  }

  /** calculates the rgb values */
  private void calculateRGB(int[] wholeScaleValues)
  {
    for (int i = 0; i != wholeScaleValues.length; i++) {
      int[] rgb = new int[3];
      rgb[0] = this.calculateR(wholeScaleValues[i]);
      rgb[1] = this.calculateG(wholeScaleValues[i]);
      rgb[2] = this.calculateB(wholeScaleValues[i]);
      channels.add(rgb);
    }
  }
  
  /**
   * calculates the correspongin red value
   * @param wholeScaleValue the value on the flat whole scale
   * @return the R value
   */
  private int calculateR(int wholeScaleValue)
  {
    int red = rgbMinValue;
    // section 1
    if (wholeScaleValue < sectionStep) {
      red = wholeScaleValue + rgbMinValue;
    }
    // section 2
    if ((wholeScaleValue > sectionStep)
        && (wholeScaleValue < (sectionStep * 2))) {
      red = rgbMaxValue;
    }
    // section 3
    if ((wholeScaleValue > (sectionStep * 2))
        && (wholeScaleValue < (sectionStep * 3))) {
      red = rgbMaxValue - (wholeScaleValue - (sectionStep * 2));
    }
    // section 4 & 5
    if ((wholeScaleValue > (sectionStep * 3))
        && (wholeScaleValue < (sectionStep * 5))) {
      red = rgbMinValue;
    }
    // section 6
    if ((wholeScaleValue > (sectionStep * 5))
        && (wholeScaleValue < (sectionStep * 6))) {
      red = wholeScaleValue - (sectionStep * 5) + rgbMinValue;
    }
    // section 7
    if (wholeScaleValue > (sectionStep * 6)) {
      red = rgbMaxValue;
    }
    return red;
  }

  /**
   * calculates the corresponding green value
   * @param wholeScaleValue the value on the flat whole scale
   * @return the G value
   */
  private int calculateG(int wholeScaleValue)
  {
    int green = rgbMinValue;
    // section 1
    if (wholeScaleValue < sectionStep) {
      green = rgbMinValue;
    }
    // section 2
    if ((wholeScaleValue > sectionStep)
        && (wholeScaleValue < (sectionStep * 2))) {
      green = wholeScaleValue - sectionStep + rgbMinValue;
    }
    // section 3 & 4
    if ((wholeScaleValue > (sectionStep * 2))
        && (wholeScaleValue < (sectionStep * 4))) {
      green = rgbMaxValue;
    }
    // section 5
    if ((wholeScaleValue > (sectionStep * 4))
        && (wholeScaleValue < (sectionStep * 5))) {
      green = rgbMaxValue - (wholeScaleValue - (sectionStep * 4));
    }
    // section 6 & 7
    if (wholeScaleValue > (sectionStep * 5)) {
      green = rgbMinValue;
    }
    return green;
  }

  /**
   * calculates the corresponding blue value 
   * @param wholeScaleValue the value on the flat whole scale
   * @return the B value
   */
  private int calculateB(int wholeScaleValue)
  {
    int blue = rgbMinValue;
    // section 1 & 2 & 3
    if (wholeScaleValue < (sectionStep * 3)) {
      blue = rgbMinValue;
    }
    // section 4
    if ((wholeScaleValue > (sectionStep * 3))
        && (wholeScaleValue < (sectionStep * 4))) {
      blue = wholeScaleValue - (sectionStep * 3) + rgbMinValue;
    }
    // section 5 & 6
    if ((wholeScaleValue > (sectionStep * 4))
        && (wholeScaleValue < (sectionStep * 6))) {
      blue = rgbMaxValue;
    }
    // section 7
    if (wholeScaleValue > (sectionStep * 6)) {
      blue = rgbMaxValue - (wholeScaleValue - (sectionStep * 6));
    }
    return blue;
  }
  
  /**
   * @return the rgb values for the compared channels
   */
  public Vector getComparedChannelsNumbers()
  {
    return comparedChannelsNumbers;
  }

  /**
   * @param comparedChannelsNumbers rgb values for the compared channels
   */
  public void setComparedChannelsNumbers(Vector comparedChannelsNumbers)
  {
    this.comparedChannelsNumbers = comparedChannelsNumbers;
  }

}
