package at.tugraz.genome.maspectras.chromaviewer;

import at.tugraz.genome.maspectras.quantification.*;
import at.tugraz.genome.maspectras.quantification.vos.*;

public class CgAreaDeterminer
{
	private CgChromatogramVO 			m_cg;
	private float						m_retTime;
	private int							m_method;
	private int							m_scanCount;
	    private CgChromatogramPointVO[]		m_pt;
    private int							m_peak;
    private int							m_upValley, m_loValley;
    
    private float						m_peakValue;
    private float						m_average;
    
    private float						m_background;
    private float						m_area;
    private float						m_areaErr;
    private float						m_peakTime;
    private float						m_peakTimeAverage;
    
    private boolean						m_good;

    
	public CgAreaDeterminer(CgChromatogramVO cg, float retTime, int method)
	{
		m_cg = cg;
		m_retTime = retTime;
		m_pt = new CgChromatogramPointVO[m_cg.getChromatogramPointVOs().size()];
		for (int i=0; i<m_cg.getChromatogramPointVOs().size(); i++)
		{
			m_pt[i] = (CgChromatogramPointVO)(m_cg.getChromatogramPointVOs().get(i));
		}
		m_cg.getChromatogramPointVOs().copyInto(m_pt);
		m_scanCount = m_pt.length;
		m_method = method;
		getMaximumAndAverage();
		findPeak();
		getBackground(50);
		getAreaAndTime();
	}
	
	public CgProbe getCgProbe()
	{
		CgProbe cx = new CgProbe(0, 0);
		
		cx.Scan = m_peak;							// Index of Peak Value
		
		cx.Charge = 0;								// TBD outwards
		cx.LowerMzBand = 0;							// TBD outwards
		cx.Mz = 0;									// TBD outwards
		cx.UpperMzBand = 0;							// TBD outwards
		
		cx.LowerValley      = m_pt[m_loValley].getScan();
		cx.Peak             = m_pt[m_peak].getScan();
		cx.PeakAmplitudeRaw = m_pt[m_peak].getIntensity();
		cx.PeakAmplitudeFit = m_pt[m_peak].getSmoothedIntensity();
		cx.UpperValley      = m_pt[m_upValley].getScan();

		cx.Background       = m_background;
		cx.Area             = m_area;
		cx.AreaError        = m_areaErr;
		
		if (m_good) cx.AreaStatus = CgAreaStatus.OK;
		else        cx.AreaStatus = CgAreaStatus.TooSmall;

		return cx;
	}
		
	private void findPeak()
    {
        int right, left;
        int i;
        int position;
        int peak;
        int direction;
         
        
        position = 0;
        while (m_pt[position].getScan()<m_retTime) position++;
        
        // =============================================================
        // determine slope on the right hand side 
        // =============================================================
        
        i = 1;
        while(position+i<m_scanCount && m_pt[position].getSmoothedIntensity()==m_pt[position+i].getSmoothedIntensity()) i++;
        if (position+i>=m_scanCount)                                                             right = 0;
        else if (m_pt[position].getSmoothedIntensity()< m_pt[position+i].getSmoothedIntensity()) right = 1;
        else                                                                                     right = -1;
        
        // =============================================================
        // determine slope on the left hand side
        // =============================================================
        
        i = -1;
        while(position+i >= 0 && m_pt[position].getSmoothedIntensity() == m_pt[position+i].getSmoothedIntensity()) i--;
        if (position+i < 0)                                                                     left = 0;
        else if (m_pt[position].getSmoothedIntensity()<m_pt[position+i].getSmoothedIntensity()) left = 1;
        else                                                                                    left = -1;
        
        if((right==-1 && left==-1) || (right==0 && left==0)) 
        {
            m_peak = position;
            getValleys();
            return;
        }
        else if (right==-1 && left==0)                       
        {
        	m_peak = 0;
            getValleys();
            return;
        }
        else if (right==0 && left==-1)
        {
        	m_peak = m_scanCount-1;
            getValleys();
            return;
        }
        
        // =============================================================
        // determine "direction"
        // =============================================================
        
        direction = -1;
        if (right<=0 && left==1)      direction = -1;
        else if (right==1 && left<=0) direction = 1;
        
        // =============================================================
        // search for peak
        // =============================================================
        
        while (position+direction >= 0 && position+direction < m_scanCount
                && m_pt[position].getSmoothedIntensity() <= m_pt[position+direction].getSmoothedIntensity()) 
        {
            position += direction;
        }
        
        // =============================================================
        // return index for peak
        // =============================================================
        
        if (position+direction < 0)                 m_peak = 0;
        else if (position+direction >= m_scanCount) m_peak = m_scanCount - 1;
        else                                        m_peak = position;
        
        getValleys();
    }
	
	private void getValleys()
	{
		if      (m_method==0) getValleysOriginal();
		else if (m_method==1) getValleysEnhanced();
	    else                  getValleysOriginal();
	}
	
	private void getValleysOriginal()
    {
        int position;
        
        // =============================================================
        // Do the upwards direction:
        // =============================================================
        
        position = m_peak;
       	while(position<m_scanCount-1)
        {
            if (m_pt[position+1].getSmoothedIntensity()>m_pt[position].getSmoothedIntensity()) break;
            position++;
        }
        
        m_upValley = position;
        if (m_upValley>m_scanCount-2) m_upValley = m_scanCount-2;
        
        // =============================================================
        // Do the downwards direction:
        // =============================================================
        
        position = m_peak;
        while(position>0)
        {
            if (m_pt[position-1].getSmoothedIntensity()>m_pt[position].getSmoothedIntensity()) break;
            position--;
        }
        m_loValley = position;
        if (m_loValley<1) m_loValley = 1;
    }

    private void getValleysEnhanced()
    {
        int		position;
        float	localPeak;
        
        
        localPeak = m_pt[m_peak].getSmoothedIntensity();
        
        // =============================================================
        // Do the upwards direction:
        // =============================================================
        
        position = m_peak;
        while(position<m_scanCount-1)
        {
            if (m_pt[position].getSmoothedIntensity()>(0.1 * localPeak) && localPeak>(10*m_average))
            {
                position++;
                continue;
            }
            if (m_pt[position+1].getSmoothedIntensity()>m_pt[position].getSmoothedIntensity()) break;
            position++;
        }
        m_upValley = position;
        if (m_upValley>m_scanCount-2) m_upValley = m_scanCount-2;
        
        // =============================================================
        // Do the downwards direction:
        // =============================================================
        
        position = m_peak;
        while(position>0)
        {
            if (m_pt[position].getSmoothedIntensity()>(0.1f * localPeak) && localPeak>(10*m_average)) 
            {
                position--;
                continue;
            }
            if (m_pt[position-1].getSmoothedIntensity()>m_pt[position].getSmoothedIntensity()) break;
            position--;
        }
        // if (position==0) return;
        m_loValley = position;
        if (m_loValley<1) m_loValley = 1;
    }

    private void getMaximumAndAverage()
    {
        int			i;
        
        m_peakValue = 0;
        m_average = 0;
        
        for (i=0; i<m_scanCount; i++)
        {
            if (m_pt[i].getSmoothedIntensity()>m_peakValue) m_peakValue = m_pt[i].getSmoothedIntensity();
            m_average += m_pt[i].getSmoothedIntensity();
        }
        m_average /= m_scanCount;
    }
    
    public void getBackground(int range)
    {
      m_background = CgAreaDeterminer.getBackground(m_pt,range,m_loValley,m_upValley);
/*        float noise, background, ave;
        int count1, count2;
        int dnRange, upRange;
        int i;
        
        // =============================================================
        // Set the background scan range
        // =============================================================
        
        dnRange = m_loValley - range;
        if (dnRange<0) dnRange = 0;
        upRange = m_upValley + range;
        if (upRange>=m_scanCount) upRange = m_scanCount-1;
        
        // =============================================================
        // get noise level
        // =============================================================
        
        noise = 0;
        count1 = 0;
        for (i = dnRange; i <= upRange; ++i)
        {
            if (m_pt[i].getIntensity() > 0) 
            {
                noise += (m_pt[i].getIntensity() - m_pt[i].getSmoothedIntensity())
                       * (m_pt[i].getIntensity() - m_pt[i].getSmoothedIntensity());
                count1++;
            }
        }
        if(count1 > 0) noise = (float)Math.sqrt(noise / count1);
        else           noise = 1;
        
        // =============================================================
        // get background
        // =============================================================
        
        background = 0;
        ave = noise;
        while(Math.abs(background - ave)>(0.01f * noise)) 
        {
            background = ave;
            ave = 0;
            count1 = 0;
            count2 = 0;
            for (i=dnRange; i<=upRange; ++i)
            {
                if(i<m_loValley || i>m_upValley) 
                {
                    if (m_pt[i].getIntensity()>0 && m_pt[i].getIntensity()<(background + noise)) 
                    {
                        ave += m_pt[i].getIntensity();
                        count1++;
                    }
                    else if (m_pt[i].getIntensity() >= (background + noise)) count2++;
                }
            }
            
            if(count1>0)       ave /= count1;
            else if(count2>0)  ave = background + 0.2f * noise;
            else               ave = 0;
        }
        m_background = ave;*/
    }
    
    public static float getBackground (CgChromatogramPointVO[] m_pt,int range, int lowerValley, int upperValley){
      float noise, background, ave;
      int count1, count2;
      int dnRange, upRange;
      int i;
      
      // =============================================================
      // Set the background scan range
      // =============================================================
      
      dnRange = lowerValley - range;
      if (dnRange<0) dnRange = 0;
      upRange = upperValley + range;
      if (upRange>=m_pt.length) upRange = m_pt.length-1;
      
      // =============================================================
      // get noise level
      // =============================================================
      
      noise = 0;
      count1 = 0;
      for (i = dnRange; i <= upRange; ++i)
      {
          if (m_pt[i].getIntensity() > 0) 
          {
              noise += (m_pt[i].getIntensity() - m_pt[i].getSmoothedIntensity())
                     * (m_pt[i].getIntensity() - m_pt[i].getSmoothedIntensity());
              count1++;
          }
      }
      if(count1 > 0) noise = (float)Math.sqrt(noise / count1);
      else           noise = 1;
      
      // =============================================================
      // get background
      // =============================================================
      
      background = 0;
      ave = noise;
      while(Math.abs(background - ave)>(0.01f * noise)) 
      {
          background = ave;
          ave = 0;
          count1 = 0;
          count2 = 0;
          for (i=dnRange; i<=upRange; ++i)
          {
              if(i<lowerValley || i>upperValley) 
              {
                  if (m_pt[i].getIntensity()>0 && m_pt[i].getIntensity()<(background + noise)) 
                  {
                      ave += m_pt[i].getIntensity();
                      count1++;
                  }
                  else if (m_pt[i].getIntensity() >= (background + noise)) count2++;
              }
          }
          
          if(count1>0)       ave /= count1;
          else if(count2>0)  ave = background + 0.2f * noise;
          else               ave = 0;
      }
      return ave;      
    }

    public void getAreaAndTime()
    {
        float rawArea, fitArea;
        float rawVal,  fitVal;
        float maxVal, ave, areaErr;
        int dnIndex, upIndex;
        int count;
        int i;
        
        m_good = true;
        
        // =============================================================
        // Get Maximum
        // =============================================================
        
        maxVal = 0;
        for (i = m_loValley; i <= m_upValley; ++i) 
        {
            if (maxVal<m_pt[i].getSmoothedIntensity()) maxVal = m_pt[i].getSmoothedIntensity();
        }
        
        // =============================================================
        // Get the Peak Area
        // =============================================================
        
        rawArea = 0;
        fitArea = 0;
        areaErr = 0;
        
        for (i = m_loValley; i <= m_upValley; i++) 
        {
            rawVal = (m_pt[i+1].getIntensity() + m_pt[i].getIntensity()) / 2 - m_background;
            if (rawVal<0) rawVal = 0;
            rawArea += rawVal * (m_pt[i+1].getScan() - m_pt[i].getScan());
            
            fitVal = (m_pt[i+1].getSmoothedIntensity() + m_pt[i].getSmoothedIntensity()) / 2 - m_background;
            if (fitVal<0) fitVal = 0;
            fitArea += fitVal * (m_pt[i+1].getScan() - m_pt[i].getScan());
            
            areaErr += 0.5f * (rawVal - fitVal) * (rawVal - fitVal)
            * (m_pt[i+1].getScan() - m_pt[i].getScan()) 
            * (m_pt[i+1].getScan() - m_pt[i].getScan());
        }
        
        m_area = (fitArea + rawArea) / 2;
        
        // =============================================================
        // The Area calculates is bad in case that:
        // - the raw area is <= 0
        // - the fitted area is <= 0
        // - the calculated area is smaller than the difference between
        //   the raw and fitted area
        // - the maximum intensity is smaller than half of the backgnd
        // =============================================================
        
        m_areaErr = (float)Math.sqrt(areaErr);
        
        if(rawArea<=0 || fitArea<=0 || m_area< m_areaErr || maxVal < 2 * m_background) 
        {
        	m_good = false;
        }
        
        // =============================================================
        // find peakTime 
        // =============================================================
        
        m_peakTime = m_pt[m_peak].getScan();
        
        // find error in peakTime
        
        float threshold = (m_pt[m_peak].getSmoothedIntensity() + m_background) / 2;
        dnIndex = m_peak;
        while(dnIndex >= m_loValley)
        {
            if (m_pt[dnIndex].getSmoothedIntensity() <= threshold) break;
            dnIndex--;
        }
        
        upIndex = m_peak;
        while(upIndex <= m_upValley)
        {
            if (m_pt[upIndex].getSmoothedIntensity() <= threshold) break;
            upIndex++;
        }
        
        if (upIndex<=dnIndex) m_peakTimeAverage = 0;
        else                  m_peakTimeAverage = (m_pt[upIndex].getScan() - m_pt[dnIndex].getScan()) / 2;
        
        
        if (m_area==0 || maxVal>=m_background/2) return;
        
        // =============================================================
        // Double check on bad data:
        // =============================================================
        
        count = 0;
        for (i=dnIndex; i<=upIndex; i++)
        {
            if(m_pt[i].getIntensity()<m_background) count++;
        }
        if(count>(upIndex-dnIndex+1)/4) 
        {
            count = 0;
            if (m_pt[dnIndex].getSmoothedIntensity()<m_pt[upIndex].getSmoothedIntensity()) ave = m_pt[dnIndex].getSmoothedIntensity();
            else                                                                           ave = m_pt[upIndex].getSmoothedIntensity();
            
            for (i=dnIndex; i<=upIndex; i++)
            {
                if (m_pt[i].getIntensity()>ave) count++;
            }
            if (count<(upIndex-dnIndex+1)/2) m_good = false;
        }
    }

}
