package at.tugraz.genome.maspectras.graphics;

import java.awt.Color;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.GradientPaint;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.image.BufferedImage;
import java.math.BigDecimal;
import java.util.Hashtable;
import java.util.Vector;

import at.tugraz.genome.maspectras.utils.Calculator;

public class HeatMapGenerator {

  private static final long serialVersionUID = -1791509019119318692L;

  public final static int NUMBER_OF_COLOR_GROUPS = 9;
  Color borderColor_=new Color(0,0,0);
  
  protected transient BufferedImage expressionImage;
  protected Vector<String> sampleNames_;
  protected Vector<String> proteinNames_;
  
  protected int defaultNrOfGradientPixels_ = 10000;

  protected final static Color DEFAULT_COLOR0 = new Color(000,040,180);
  protected final static Color DEFAULT_COLOR1 = new Color(000,153,102);
  protected final static Color DEFAULT_COLOR2 = new Color(255,255,000);
  protected final static Color DEFAULT_COLOR3 = new Color(255,255,000);
  protected final static Color DEFAULT_COLOR4 = new Color(255,000,000);
  
  protected float minValue_;
  protected float maxValue_;
  protected BufferedImage gradient_;
  
  protected int heatRectWidth_ = 20;
  protected int heatRectHeight_ = 10;
  protected int borderCorrection_ = 1;
  
  protected int annotationSpace_ = 200;
  protected int sampleSpace_ = 80;
  protected int pictureIndent_ = 5;
  protected int gradientHeight_ = 10;
  protected int proteinNameSpace_ = 3;
  
  protected int proteinTextWidthMax_ = 0;
  protected int sampleNameStart_ = 0;
  
  protected Hashtable<String,Hashtable<String,Color>> attentionValues_;
  
  int partCorrection_ = 0;
  
  float[][] data_;
  
  public HeatMapGenerator(float[][] data, Vector<String> sampleNames, Vector<String> proteinNames)
  {
    this.setInputValues(data, sampleNames, proteinNames, new Hashtable<String,Hashtable<String,Color>>());
    this.init();
  }

  public HeatMapGenerator(float[][] data, Vector<String> sampleNames, Vector<String> proteinNames, Hashtable<String,Hashtable<String,Color>> attentionValues)
  {
    this.setInputValues(data, sampleNames, proteinNames, attentionValues);
    this.init();
  }
  
  private void setInputValues(float[][] data, Vector<String> sampleNames, Vector<String> proteinNames, Hashtable<String,Hashtable<String,Color>> attentionValues){
    this.data_ = data;
    this.sampleNames_ = sampleNames;
    this.proteinNames_ = proteinNames;   
    this.attentionValues_ = attentionValues;
  }
  
  public void init() {
    this.initGradient();
 }
  
  private void initGradient(){
    this.calculateMinMax();
    //gradient_ = HeatMapGenerator.createDefaultGradientImage(this.defaultNrOfGradientPixels_);
    gradient_ = this.createThreeColGradientImage(this.defaultNrOfGradientPixels_);
    
  }
  
  private void calculateMinMax(){
    minValue_ = Float.MAX_VALUE;
    maxValue_ = Float.MIN_VALUE;
    for (int i=0; i!=data_.length;i++){
      for (int j=0; j!=data_[0].length;j++){
        if (data_[i][j]<minValue_)
          minValue_ = data_[i][j];
        if (data_[i][j]>maxValue_)
          maxValue_ = data_[i][j];
      }
    }
    minValue_ = Calculator.roundFloat(minValue_,0,BigDecimal.ROUND_DOWN);
    maxValue_ = Calculator.roundFloat(maxValue_,0,BigDecimal.ROUND_UP);
  }
  
  public BufferedImage createImage() {
    return this.createImage(null);
  }
  
  public BufferedImage createImage(Graphics2D extGraphics) {
    BufferedImage dummyImage = new BufferedImage(1000,1000,BufferedImage.TYPE_3BYTE_BGR);
    Font descriptionFont = new Font("Dialog",Font.PLAIN, 9);
    Graphics2D dummyGraphics = (Graphics2D)dummyImage.getGraphics();
    dummyGraphics.setFont(descriptionFont);
    FontMetrics descriptionFontMetrics = dummyGraphics.getFontMetrics();
    int textWidth = 0;
    for (String sampleName : sampleNames_){
      String  toDisplay = getSampleNameToDisplay(sampleName);
      int width = descriptionFontMetrics.stringWidth(toDisplay);
      if (width>textWidth){
        textWidth = width;
      }  
    }
    int annotationWidth = 0;
    for (String proteinName : proteinNames_){
      int width = descriptionFontMetrics.stringWidth(proteinName);
      if (width>annotationWidth){
        annotationWidth = width;
      }  
    }
    try {
      this.sampleNameStart_ = this.pictureIndent_+this.gradientHeight_+descriptionFontMetrics.getHeight();
      this.sampleSpace_ = textWidth+sampleNameStart_;
      this.annotationSpace_ = annotationWidth+2*this.pictureIndent_;
      Graphics2D expressionGraphics;
      int xcorrection=0;
      int ycorrection=0;
      int xSpaceForMap = this.heatRectWidth_*this.sampleNames_.size()+borderCorrection_;
      int imageSizeX = xSpaceForMap+/*(this.getNumberOfParts()-1)*10*/+xcorrection+annotationSpace_+2*pictureIndent_;
      expressionImage=new BufferedImage(imageSizeX,
                                         this.heatRectHeight_*(this.proteinNames_.size())+borderCorrection_+ycorrection+sampleSpace_+2*pictureIndent_,
                                         BufferedImage.TYPE_3BYTE_BGR);
       expressionGraphics=(Graphics2D)expressionImage.getGraphics();
       if (extGraphics!=null)
         expressionGraphics=extGraphics;
       expressionGraphics.setColor(getBackgroundColor());
       expressionGraphics.fillRect(0, 0, expressionImage.getWidth(), expressionImage.getHeight());
       this.paintHeader(expressionGraphics, imageSizeX);
       this.paintSampleNames(expressionGraphics,this.pictureIndent_ /*imageSizeX-pictureIndent_-xSpaceForMap-1*/,sampleSpace_);
       paintExpressionImage(expressionGraphics, this.pictureIndent_/*imageSizeX-pictureIndent_-xSpaceForMap-1*/, sampleSpace_/*, geneOrder, sampleOrder*/);

       this.paintProteinNames(expressionGraphics,this.pictureIndent_+xSpaceForMap,sampleSpace_);
       
       } catch (OutOfMemoryError e) {
       }
       return expressionImage;      
 }
  
  
  protected void paintHeader(Graphics2D expressionGraphics,int imageSizeX){
    //paints the gradient
//    GradientPaint gp1 = new GradientPaint(this.pictureIndent_, 0, HeatMapGenerator.DEFAULT_COLOR1, (imageSizeX-1)/2, 0,HeatMapGenerator.DEFAULT_COLOR2);
//    GradientPaint gp2 = new GradientPaint((imageSizeX-1)/2, 0, HeatMapGenerator.DEFAULT_COLOR3,  imageSizeX-2*this.pictureIndent_, 0, HeatMapGenerator.DEFAULT_COLOR4);
    GradientPaint gp0 = new GradientPaint(this.pictureIndent_, 0, HeatMapGenerator.DEFAULT_COLOR0, this.pictureIndent_+(imageSizeX-1-2*this.pictureIndent_)/3, 0,HeatMapGenerator.DEFAULT_COLOR1);

    GradientPaint gp1 = new GradientPaint(this.pictureIndent_+(imageSizeX-1-2*this.pictureIndent_)/3, 0, HeatMapGenerator.DEFAULT_COLOR1, this.pictureIndent_+(2*(imageSizeX-1-2*this.pictureIndent_)/3), 0,HeatMapGenerator.DEFAULT_COLOR2);
   GradientPaint gp2 = new GradientPaint(this.pictureIndent_+(2*(imageSizeX-1-2*this.pictureIndent_)/3), 0, HeatMapGenerator.DEFAULT_COLOR3,  imageSizeX-2*this.pictureIndent_, 0, HeatMapGenerator.DEFAULT_COLOR4);
    int gradientYPosition = this.pictureIndent_;
    expressionGraphics.setPaint(gp0);
    expressionGraphics.fillRect(this.pictureIndent_, gradientYPosition, (imageSizeX-2*this.pictureIndent_)/3, gradientHeight_);
    expressionGraphics.setPaint(gp1);
    expressionGraphics.fillRect(this.pictureIndent_+(imageSizeX-1-2*this.pictureIndent_)/3, gradientYPosition, (imageSizeX-2*this.pictureIndent_)/3, gradientHeight_);
 //   expressionGraphics.fillRect(this.pictureIndent_, gradientYPosition, (imageSizeX-1-2*this.pictureIndent_)/2, gradientHeight_);
    expressionGraphics.setPaint(gp2);
    expressionGraphics.fillRect(this.pictureIndent_+(2*(imageSizeX-1-2*this.pictureIndent_)/3), gradientYPosition, (imageSizeX-2*this.pictureIndent_)/3, gradientHeight_);
//    expressionGraphics.fillRect((imageSizeX-1)/2, gradientYPosition, (imageSizeX-1-2*this.pictureIndent_)/2, gradientHeight_);
    
    //paints the text of the gradient
    String minText = String.valueOf(this.minValue_);
    String middleText = String.valueOf((this.maxValue_-this.minValue_)/2f);
    String maxText = String.valueOf(this.maxValue_);
    Font descriptionFont = new Font("Dialog",Font.PLAIN, 8);
    expressionGraphics.setColor(getLegendColor());
    expressionGraphics.setFont(descriptionFont);
    FontMetrics descriptionFontMetrics = expressionGraphics.getFontMetrics();
    int textHeight = descriptionFontMetrics.getHeight();
    int textWidth = descriptionFontMetrics.stringWidth(minText);
    int textYPosition = gradientYPosition+gradientHeight_+textHeight/2+2;
    expressionGraphics.drawString(minText,this.pictureIndent_,textYPosition);
    textWidth = descriptionFontMetrics.stringWidth(middleText);
    expressionGraphics.drawString(middleText,imageSizeX/2-textWidth/2,textYPosition);
    textWidth = descriptionFontMetrics.stringWidth(maxText);
    expressionGraphics.drawString(maxText,imageSizeX-this.pictureIndent_-textWidth,textYPosition);
    
  }
  

  private void paintExpressionImage(Graphics2D expressionGraphics, int xoffset,
      int yoffset)
  {
    float dummyfloat;
    expressionGraphics.setColor(borderColor_);
//    expressionGraphics.fillRect(xoffset, yoffset, expressionImage.getWidth(),
//        expressionImage.getHeight());
    

    for (int i = 0; i < this.proteinNames_.size(); i++) {
      partCorrection_ = 0;
      for (int j = 0; j < this.sampleNames_.size(); j++) {
        dummyfloat = this.getValue(j, i);
        Color dummycolor = new Color(getColorForValue(dummyfloat));
        expressionGraphics.setColor(dummycolor);
        expressionGraphics.fillRect(this.heatRectWidth_ * j + 1
            + xoffset + partCorrection_, this.heatRectHeight_ * i + 1
            + yoffset, this.heatRectWidth_ - 1, this.heatRectHeight_ - 1);
        if (attentionValues_.containsKey(sampleNames_.get(j)) && attentionValues_.get(sampleNames_.get(j)).containsKey(proteinNames_.get(i))){
          paintAttentionRectangle(expressionGraphics,attentionValues_.get(sampleNames_.get(j)).get(proteinNames_.get(i)),i,j,xoffset,yoffset);
        }              
      }
    }
  }

  
  /**
   * this method paints an unfilled rectangle around the heat map rectangle containing the expression color
   * @param g2 Graphics2 object for painting
   * @param color the color to be used for painting the unfilled rectangle
   * @param row the row number in the expression image
   * @param column the column number in the expression image
   * @param xoffset the offset in x direction of the heat map within the image
   * @param yoffset the offset in y direction of the heat map within the image
   */
  public void paintAttentionRectangle(Graphics2D g2, Color color, int row, int column, int xoffset, int yoffset){
    g2.setColor(attentionValues_.get(sampleNames_.get(column)).get(proteinNames_.get(row)));
    g2.drawRect(this.heatRectWidth_ * column + 1
        + xoffset + partCorrection_, this.heatRectHeight_ * row + 1
        + yoffset, this.heatRectWidth_ - 2, this.heatRectHeight_ - 2);
  }

  
  private void paintSampleNames(Graphics2D expressionGraphics, int xoffset, int sampleSpace_){
    Font bigFont = new Font("SansSerif", Font.PLAIN, 200);
    expressionGraphics.setFont(bigFont);
    FontMetrics fm2 = expressionGraphics.getFontMetrics();

    Font descriptionFont = new Font("Dialog",Font.PLAIN, 9);
    expressionGraphics.setColor(getLegendColor());
    
    expressionGraphics.setFont(descriptionFont);
    FontMetrics descriptionFontMetrics = expressionGraphics.getFontMetrics();
    int textHeight = descriptionFontMetrics.getHeight();
    double rotationAngle = -Math.PI/2.0;
    if (!isOSMacAndJavaLookAndFeel())
      expressionGraphics.rotate(rotationAngle);
    for (int i=0;i!=this.sampleNames_.size();i++){
      String sampleName = sampleNames_.get(i);
      String toDisplay = this.getSampleNameToDisplay(sampleName);
      if (isOSMacAndJavaLookAndFeel()){       
        BufferedImage image = new BufferedImage(fm2.stringWidth(toDisplay), fm2.getHeight(),
            BufferedImage.TYPE_INT_ARGB);
        Graphics2D g22 = (Graphics2D) image.getGraphics();
        g22.setColor(getLegendColor());
        g22.setFont(bigFont);
        g22.drawString(toDisplay,0,fm2.getAscent());
        int sw2 = fm2.stringWidth(toDisplay);
        int swXExtension2 = (int)(((double)sw2)*Math.cos(-rotationAngle));
        int swYExtension2 = (int)(((double)sw2)*Math.sin(-rotationAngle));
        int shYExtension2 = (int)(((double)fm2.getHeight())*Math.cos(-rotationAngle));
        int shXExtension2 = (int)(((double)fm2.getHeight())*Math.sin(-rotationAngle));
      
        BufferedImage image2 = new BufferedImage(swXExtension2+shXExtension2, swYExtension2+shYExtension2,
            BufferedImage.TYPE_INT_ARGB);
        Graphics2D g23 = (Graphics2D) image2.getGraphics();
        g23.rotate(rotationAngle);
        int normalStringXCoordinate2 = 0;
        int normalStringYCoordinate2 = swYExtension2;
        double coordinateRadius2 = Math.sqrt(Math.pow(0, 2)+Math.pow(normalStringYCoordinate2, 2));
        double angle2 = Math.atan((double)normalStringYCoordinate2/(double)normalStringXCoordinate2);
        g23.drawImage(image,(int)(coordinateRadius2*Math.cos(angle2-rotationAngle)),(int)(coordinateRadius2*Math.sin(angle2-rotationAngle)),null);
        g23.rotate(-rotationAngle);   
        //expressionGraphics.drawImage(image2.getScaledInstance(descriptionFontMetrics.getHeight(), descriptionFontMetrics.stringWidth(toDisplay), 0),this.heatRectWidth_ * i +this.heatRectWidth_/2,sampleSpace_-2-descriptionFontMetrics.stringWidth(toDisplay),null);
        image2 = MacOSResizer.resizeTrick(image2, descriptionFontMetrics.getHeight(), descriptionFontMetrics.stringWidth(toDisplay));
        expressionGraphics.drawImage(image2,this.heatRectWidth_ * i +this.heatRectWidth_/2,sampleSpace_-2-descriptionFontMetrics.stringWidth(toDisplay),null);
      }else{
        expressionGraphics.drawString(toDisplay,-sampleSpace_+2,this.heatRectWidth_ * i +this.heatRectWidth_/2+textHeight/2
            + xoffset);
      }
    }
    if (!isOSMacAndJavaLookAndFeel())
      expressionGraphics.rotate(-rotationAngle);

  }
  
  private void paintProteinNames(Graphics2D expressionGraphics, int xoffset, int yoffset){
    Font descriptionFont = new Font("Dialog",Font.PLAIN, 9);
    expressionGraphics.setColor(getLegendColor());
    expressionGraphics.setFont(descriptionFont);
    FontMetrics descriptionFontMetrics = expressionGraphics.getFontMetrics();
    int textHeight = descriptionFontMetrics.getHeight();
    for (int i=0;i!=this.proteinNames_.size();i++){
      String proteinName = this.proteinNames_.get(i);
      int textWidth = descriptionFontMetrics.stringWidth(proteinName);
      if (textWidth>0)
        proteinTextWidthMax_ = textWidth;
      expressionGraphics.drawString(proteinName,xoffset+proteinNameSpace_,(this.heatRectHeight_*i) + 2 + yoffset+textHeight/2);
    }
  }
  
  
  protected int getColorForValue(float value){
    float scale = this.getColorScale();     
    int rgb1 = this.gradient_.getRGB((int)((value-this.minValue_)*scale), 0);
    return rgb1;
    //return this.mapColorToScale(rgb1);
  }
  
  protected float[] mapColorToScale(int rgb1){
    float[] color = new float[3];
    color[0] = (float)((rgb1 >> 16) & 0xFF)/255f;
    color[1] = (float)((rgb1 >>  8) & 0xFF)/255f;
    color[2] = (float)((rgb1 >>  0) & 0xFF)/255f;
    return color;    
  }
  
  protected BufferedImage createThreeColGradientImage(int sizeX) {
    BufferedImage gradient = new BufferedImage(sizeX, 1, BufferedImage.TYPE_3BYTE_BGR);
    Graphics2D graphics = gradient.createGraphics();
    //protected final static Color DEFAULT_COLOR0 = new Color(000,010,150);
    GradientPaint gp0 = new GradientPaint(0, 0, HeatMapGenerator.DEFAULT_COLOR0, (sizeX-1)/3, 0,HeatMapGenerator.DEFAULT_COLOR1);
//    GradientPaint gp1 = new GradientPaint(0, 0, HeatMapGenerator.DEFAULT_COLOR1, (sizeX-1)/15, 0,HeatMapGenerator.DEFAULT_COLOR2);
//    GradientPaint gp2 = new GradientPaint((sizeX-1)/15, 0, HeatMapGenerator.DEFAULT_COLOR3,  sizeX-1, 0, HeatMapGenerator.DEFAULT_COLOR4);
    GradientPaint gp1 = new GradientPaint((sizeX-1)/3, 0, HeatMapGenerator.DEFAULT_COLOR1, 2*((sizeX-1)/3), 0,HeatMapGenerator.DEFAULT_COLOR2);
    GradientPaint gp2 = new GradientPaint(2*((sizeX-1)/3), 0, HeatMapGenerator.DEFAULT_COLOR3,  sizeX-1, 0, HeatMapGenerator.DEFAULT_COLOR4);
    graphics.setPaint(gp0);
    graphics.drawRect(0, 0, (sizeX-1)/3, 1);
    graphics.setPaint(gp1);
    graphics.drawRect((sizeX-1)/3, 0, 2*((sizeX-1)/3), 1);
    graphics.setPaint(gp2);
    graphics.drawRect(2*((sizeX-1)/3), 0, sizeX-1, 1);
    graphics.dispose();    
    return gradient;
  }
  
  protected static BufferedImage createDefaultGradientImage(int sizeX) {
    BufferedImage gradient = new BufferedImage(sizeX, 1, BufferedImage.TYPE_3BYTE_BGR);
    Graphics2D graphics = gradient.createGraphics();
//    GradientPaint gp1 = new GradientPaint(0, 0, HeatMapGenerator.DEFAULT_COLOR1, (sizeX-1)/15, 0,HeatMapGenerator.DEFAULT_COLOR2);
//    GradientPaint gp2 = new GradientPaint((sizeX-1)/15, 0, HeatMapGenerator.DEFAULT_COLOR3,  sizeX-1, 0, HeatMapGenerator.DEFAULT_COLOR4);
    GradientPaint gp1 = new GradientPaint(0, 0, HeatMapGenerator.DEFAULT_COLOR1, (sizeX-1)/2, 0,HeatMapGenerator.DEFAULT_COLOR2);
    GradientPaint gp2 = new GradientPaint((sizeX-1)/2, 0, HeatMapGenerator.DEFAULT_COLOR3,  sizeX-1, 0, HeatMapGenerator.DEFAULT_COLOR4);
    graphics.setPaint(gp1);
    graphics.drawRect(0, 0, (sizeX-1)/2, 1);
    graphics.setPaint(gp2);
    graphics.drawRect((sizeX-1)/2, 0, sizeX-1, 1);
    graphics.dispose();    
    return gradient;
  }
  
  public float getValue(int x, int y) {
    return data_[x][y];
 }
  
  protected float getColorScale(){
    return (float)(10000-1)/(this.maxValue_-this.minValue_);
  }
  
  public int getExpressionImageXStart(){
    return this.pictureIndent_;
  }

  public int getExpressionImageXEnd(){
    return this.pictureIndent_+this.heatRectWidth_*sampleNames_.size();
  }
  
  public int getExpressionImageYStart(){
    return sampleSpace_;
  }

  public int getExpressionImageYEnd(){
    return this.sampleSpace_+this.heatRectHeight_*this.proteinNames_.size();
  }
  
  public Vector<String> getCellName(int x, int y){
    Vector<String> cellName = new Vector<String>();
    String sampleName = null;
    String proteinName = null;
    int[] cellPostion = this.getCellPosition(x, y);
    if (cellPostion[0]>=0&&cellPostion[1]>=0){
      sampleName = sampleNames_.get(cellPostion[0]);
      proteinName = proteinNames_.get(cellPostion[1]);
    }
    cellName.add(sampleName);
    cellName.add(proteinName);
    return cellName;
  }
  
  public int[] getCellPosition(int x, int y){
    int[] cellPosition = new int[2];
    cellPosition[0] = -1;
    cellPosition[1] = -1;
    if (this.getExpressionImageXStart()<=x&&x<this.getExpressionImageXEnd()&&
        this.getExpressionImageYStart()<=y&&y<this.getExpressionImageYEnd()){
      cellPosition[0] =this.calculateXPosition(x);
      cellPosition[1] = this.calculateYPosition(y);;
    }
    return cellPosition;
  }
  
  public Rectangle getRectangleForCell(int x, int y){
    Rectangle rect = null;
    if (this.getExpressionImageXStart()<=x&&x<this.getExpressionImageXEnd()&&
        this.getExpressionImageYStart()<=y&&y<this.getExpressionImageYEnd()){
      int cellColumn = this.calculateXPosition(x);
      int cellRow = this.calculateYPosition(y);
      rect = getRectangleForCellByRowAndColumn(cellRow, cellColumn);
    }
    return rect;
  }
  
  /**
   * returns a rectangle of a heat map cell defined by row and column number
   * @param cellRow the row number of the cell
   * @param cellColumn the column number of the cell
   * @return rectangle of a heat map cell defined by row and column number
   */
  public Rectangle getRectangleForCellByRowAndColumn(int cellRow, int cellColumn){
    return new Rectangle(this.heatRectWidth_ * cellColumn+getExpressionImageXStart(),
        this.heatRectHeight_ * cellRow+getExpressionImageYStart(),this.heatRectWidth_,
        this.heatRectHeight_);
  }
  
  public int getRowNameStart(){
    return this.getExpressionImageXEnd()+this.proteinNameSpace_;
  }
  
  public int getRowNameEnd(){
    return this.getRowNameStart()+this.proteinTextWidthMax_;
  }
  
  public String getRowName(int x, int y){
    String proteinName = null;
    if (this.getRowNameStart()<=x&&x<this.getRowNameEnd()&&
        this.getExpressionImageYStart()<=y&&y<this.getExpressionImageYEnd()){
      proteinName = proteinNames_.get(this.calculateYPosition(y));
    }
    return proteinName;
  }
  
  public Rectangle getRectangleForRowName(String rowName){
    int rowPosition = -1;
    int count = 0;
    for (String name : this.proteinNames_){
      if (rowName.equalsIgnoreCase(name)){
        rowPosition = count;
        break;
      }
      count++;
    }
    Rectangle rect = null;
    if (rowPosition>=0){
      rect = new Rectangle(this.getRowNameStart(),this.heatRectHeight_ * rowPosition+getExpressionImageYStart(),
          this.proteinTextWidthMax_+this.proteinNameSpace_, this.heatRectHeight_);
    }
    return rect;
  }
  
  private int calculateXPosition(int x){
    return (x-this.getExpressionImageXStart())/heatRectWidth_;
  }  
  
  private int calculateYPosition(int y){
    return (y-this.getExpressionImageYStart())/heatRectHeight_;
  }

  public int getColumnNameStart(){
    return this.sampleNameStart_-2;
  }
  
  public int getColumnNameEnd(){
    return this.sampleSpace_-2;
  }

  public String getColumnName(int x, int y){
    String sampleName = null;
    if (this.getExpressionImageXStart()<=x&&x<this.getExpressionImageXEnd()&&
        this.getColumnNameStart()<=y&&y<this.getColumnNameEnd()){
      sampleName = sampleNames_.get(this.calculateXPosition(x));
    }
    return sampleName;
  }
  
  public Rectangle getRectangleForColumnName(String columnName){
    int columnPosition = -1;
    int count = 0;
    for (String name : this.sampleNames_){
      if (columnName.equalsIgnoreCase(name)){
        columnPosition = count;
        break;
      }
      count++;
    }
    Rectangle rect = null;
    if (columnPosition>=0){
      rect = new Rectangle(this.heatRectWidth_ * columnPosition+getExpressionImageXStart(),this.getColumnNameStart(),
          this.heatRectWidth_, this.getColumnNameEnd()-this.getColumnNameStart());
    }
    return rect;
  }
  
  protected String getSampleNameToDisplay(String sampleName){
    return sampleName;
  }
  
  protected boolean isOSMacAndJavaLookAndFeel(){
    return false;
  }
  
  protected Color getLegendColor(){
    return Color.WHITE;
  }
  
  protected Color getBackgroundColor(){
    return Color.BLACK;
  }
  
  /**
   * returns the color of a heat map cell defined by row and column number
   * @param cellRow the row number of the cell
   * @param cellColumn the column number of the cell
   * @return color of a heat map cell defined by row and column number
   */
  public Color getColorForCell(int cellRow, int cellColumn){
    return new Color(getColorForValue(this.getValue(cellColumn, cellRow)));
  }
  
}
