package at.tugraz.genome.maspectras.quantification;

import java.util.Vector;



/**
 * Core class for the peak quantification. Two classes inherit from this one
 * the CgAnalyzer which makes use of the original files (mzXML,mzData or RAW)
 * and the ChromaAnalyzer which uses the new Chroma file
 * 
 * @author Juergen Hartler
 *
 */
public abstract class Analyzer
{

  /** Parameters and Results for Calculation*/
  protected CgParameterSet m_params;
  /** The calculated chromatograms */
  protected CgChromatogram m_chroma[];
  /** The amount of parts the raw file should be divided to save memory*/
  protected int numberOfCachedParts_;  
    
  /**
   * Returns the calculated area for one peptide
   * @return the total area (over several charge states)
   */
  public float getTotalArea()
  {
    return m_params.Area;
  }
  
  protected void checkDuplicate(Vector probes,CgProbe probe){
    for (int i=0;i!=probes.size();i++){
      CgProbe aProbe = (CgProbe)probes.get(i);
      if (aProbe.EqualValleys(probe) == true)
        probe.AreaStatus = CgAreaStatus.Duplicate;
      else{
        if (aProbe.isCoveredByThisProbe(probe)){
          if (aProbe.Area>probe.Area)
           probe.AreaStatus = CgAreaStatus.Duplicate;
          else
           aProbe.AreaStatus = CgAreaStatus.Duplicate;
        } else if (probe.isCoveredByThisProbe(aProbe)){
          if (aProbe.Area>probe.Area)
            probe.AreaStatus = CgAreaStatus.Duplicate;
           else
            aProbe.AreaStatus = CgAreaStatus.Duplicate;
        }  
      }
    }
  }
  
  public static int findIndexByTime(float retentionTime, CgChromatogram chrom){
    int scan = 0;
    for (int i=0; i!=chrom.ScanCount;i++){
      if (chrom.Value[i][0]>=retentionTime)
        return scan;
      scan = i;
    }
    return scan;
  }
  
  /**
   * returns the highest intensity (smoothed) of a chromatogram that is detected
   * inside found probes - used for zooming
   * @param probes the found peak identifications
   * @param chrom the chromatogram to search for
   * @return the highest intensity
   */
  public static float getHighestIntensityOfProbes(Vector<CgProbe> probes, CgChromatogram chrom){
    float highestInt = 0f;
    for (CgProbe probe : probes){
      int scan = findIndexByTime(probe.LowerValley,chrom);
      while (scan<chrom.Value.length && chrom.Value[scan][0]<probe.UpperValley){
        if (chrom.Value[scan][2]>highestInt) highestInt = chrom.Value[scan][2];
        scan++;
      }
    }
    return highestInt;
  }

  
  /**
   * Calculates the chromatograms and the areas for a peptide with
   * the corresponding input parameters
   */
  public void Process(CgParameterSet params) throws CgException
  {
    int i, j, ndx;
    boolean chromaRequest[];
    CgProbe px, py;
    CgChromatogram cx;

    m_params = params;

    // =============================================================
    // We need only the Cromatograms which are requested in the Pa-
    // rameter list. So let us determine this before.
    // =============================================================

    chromaRequest = new boolean[CgDefines.MaxCharge];
    for (i = 0; i < m_params.ProbeCount(); i++) {
      chromaRequest[m_params.Probe(i).Charge - 1] = true;
    }

    // =============================================================
    // We generate a new Chromatogram for each required charge sta-
    // te, then we go through each scan and add the Value pairs to
    // the newly generated chromatograms. Finally we smooth.
    // =============================================================
    this.readTheChromatograms();

    // =============================================================
    // Now it's time to determine the peaks, which are defined in
    // the individual Parameter Sets
    // =============================================================

    for (i = 0; i < m_params.ProbeCount(); i++) {
      px = m_params.Probe(i);
      cx = m_chroma[px.Charge - 1];

      ndx = this.findBaseScanIndex(px.Scan);
      cx.GetMaximumAndAverage();
      cx.FindPeak(ndx, 1, m_params.ValleyMethod);
      cx.GetBackground(50);
      cx.GetAreaAndTime();

      // =========================================================
      // We have results, let us transfer those into the structure
      // =========================================================
      px = this.copyResults(px, cx);
    }

    // =============================================================
    // Now we have to identify duplicates:
    // =============================================================

    for (i = 0; i < m_params.ProbeCount() - 1; i++) {
      px = m_params.Probe(i);
      for (j = i + 1; j < m_params.ProbeCount(); j++) {
        py = m_params.Probe(j);
        if (px.EqualValleys(py) == true)
          py.AreaStatus = CgAreaStatus.Duplicate;
        else{
          if (px.isCoveredByThisProbe(py))
            px.AreaStatus = CgAreaStatus.Duplicate;
          if (py.isCoveredByThisProbe(px))
            py.AreaStatus = CgAreaStatus.Duplicate;         
        }
      }
    }

    // =============================================================
    // Finally we can calculate the total area
    // =============================================================

    m_params.Area = 0;
    for (i = 0; i < m_params.ProbeCount(); i++)

    {
      if (m_params.Probe(i).AreaStatus == CgAreaStatus.OK) {
        m_params.Area += m_params.Probe(i).Area;
      }
    }
  }
  
  /**
   * copies the detected peak values from the chromatogram VO to a separate CgProbe VO
   * @param px the CgProbe VO that shall hold the final peak values
   * @param cx the chromatogram VO to copy the values from
   * @return the  the CgProbe VO that contains the detected peak values 
   */
  private static CgProbe copyResults(CgProbe px, CgChromatogram cx){
    return copyResults(px, cx, false);
  }
  
  /**
   * copies the detected peak values from the chromatogram VO to a separate CgProbe VO
   * @param px the CgProbe VO that shall hold the final peak values
   * @param cx the chromatogram VO to copy the values from
   * @param ignoreZeros shall missing data points (zero values) be ignored for detection of the 10% and 50% values
   * @return the  the CgProbe VO that contains the detected peak values 
   */
  protected static CgProbe copyResults(CgProbe px, CgChromatogram cx, boolean ignoreZeros){
    if (cx.Good){
      int highestScan = cx.UpValley+1;
      if (highestScan>cx.ScanCount) highestScan = cx.ScanCount;
      float highestIntensity = 0f;
      int rawPeakMaximum = 0;
      for (int i=cx.LoValley; i!=highestScan; i++){
        if (cx.Value[i][1]>highestIntensity){
          highestIntensity = cx.Value[i][1];
          rawPeakMaximum = i;
        }
      }
      px.setApexIntensity(cx.Value[rawPeakMaximum][1]);
      boolean tenPercSet = false;
      boolean fiftyPercSet = false;      
      float last50PercZero = -1f;
      float last10PercZero = -1f;
      
      for (int i=rawPeakMaximum; i>=cx.LoValley; i--){
        if (ignoreZeros && cx.Value[i][1]>0){
          if (last50PercZero>0) last50PercZero = -1f;
          if (last10PercZero>0) last10PercZero = -1f;
        }
        
        if (!fiftyPercSet && cx.Value[i][1]<=0.5f*cx.Value[rawPeakMaximum][1]){
          if (cx.Value[i][1]==0 && ignoreZeros){
            if (last50PercZero<0)
              last50PercZero = cx.Value[i][0];
          } else {
            px.setLowerValley50(cx.Value[i][0]);
            fiftyPercSet = true;
          }
        }
        if (!tenPercSet && cx.Value[i][1]<=0.1f*cx.Value[rawPeakMaximum][1]){
          if (cx.Value[i][1]==0 && ignoreZeros){
            if (last10PercZero<0)
              last10PercZero = cx.Value[i][0];
          } else {
            px.setLowerValley10(cx.Value[i][0]);
            tenPercSet = true;
            break;
          }
        }
      }
      if (ignoreZeros && !fiftyPercSet && last50PercZero>0){
        px.setLowerValley50(last50PercZero);
        fiftyPercSet = true;
      }
      if (ignoreZeros && !tenPercSet && last10PercZero>0){
        px.setLowerValley10(last10PercZero);
        tenPercSet = true;
      }
      if (!fiftyPercSet) px.setLowerValley50(cx.Value[cx.LoValley][0]);
      if (!tenPercSet) px.setLowerValley10(cx.Value[cx.LoValley][0]);
      
      tenPercSet = false;
      fiftyPercSet = false;
      last50PercZero = -1f;
      last10PercZero = -1f;

      for (int i=rawPeakMaximum; i<highestScan; i++){
        if (ignoreZeros && cx.Value[i][1]>0){
          if (last50PercZero>0) last50PercZero = -1f;
          if (last10PercZero>0) last10PercZero = -1f;
        }

        if (!fiftyPercSet && cx.Value[i][1]<=0.5f*cx.Value[rawPeakMaximum][1]){
          if (cx.Value[i][1]==0 && ignoreZeros){
            if (last50PercZero<0)
              last50PercZero = cx.Value[i][0];
          } else {
            px.setUpperValley50(cx.Value[i][0]);
            fiftyPercSet = true;
          }
        }
        if (!tenPercSet && cx.Value[i][1]<=0.1f*cx.Value[rawPeakMaximum][1]){
          if (cx.Value[i][1]==0 && ignoreZeros){
            if (last10PercZero<0)
              last10PercZero = cx.Value[i][0];
          } else {
            px.setUpperValley10(cx.Value[i][0]);
            tenPercSet = true;
            break;
          }
        }
      }
      if (ignoreZeros && !fiftyPercSet && last50PercZero>0){
        px.setUpperValley50(last50PercZero);
        fiftyPercSet = true;
      }
      if (ignoreZeros && !tenPercSet && last10PercZero>0){
        px.setUpperValley10(last10PercZero);
        tenPercSet = true;
      }
      if (highestScan>=cx.ScanCount) highestScan = cx.ScanCount-1;
      if (!fiftyPercSet) px.setUpperValley50(cx.Value[highestScan][0]);
      if (!tenPercSet) px.setUpperValley10(cx.Value[highestScan][0]);

    }
    px.LowerMzBand = cx.LowerMzBand;
    px.Mz = cx.Mz;
    px.UpperMzBand = cx.UpperMzBand;

    px.LowerValley = cx.Value[cx.LoValley][0];
    px.Peak = cx.Value[cx.Peak][0];
    px.PeakAmplitudeRaw = cx.Value[cx.Peak][1];
    px.PeakAmplitudeFit = cx.Value[cx.Peak][2];
    px.UpperValley = cx.Value[cx.UpValley][0];

    px.Background = cx.Background;
    px.Area = cx.Area;
    px.AreaError = cx.AreaErr;
    
    px.setHighestIntensity(cx.getHighestIntensity());

    if (cx.Good == true)
      px.AreaStatus = CgAreaStatus.OK;
    else if (!cx.anythingThere)
      px.AreaStatus = CgAreaStatus.NothingThere;
    else
      px.AreaStatus = CgAreaStatus.TooSmall;
    return px;
  }
  
  
  public CgChromatogram[] GetArea(float fromMz, float toMz, float step,
      float lowerMzBand, float upperMzBand) throws CgException
  {
    int i, n;
    CgChromatogram c[];
    float curMz;

    if (fromMz > toMz)
      return null;
    if (step <= 0)
      return null;
    // *1.005 because the integer conversion rounds always to smaller values,
    // with a step-size smaller 0.1 one chromatogram is missing
    n = (int) (((toMz - fromMz) * 1.005) / step) + 1;
    c = new CgChromatogram[n];

    // =============================================================
    // We generate a new Chromatogram for each required mz value,
    // then we go through each scan and add the Value pairs to
    // the newly generated chromatograms. Finally we smooth.
    // =============================================================

    curMz = fromMz;
    for (i = 0; i < n; i++) {
      c[i] = this.readAChromatogram(curMz, lowerMzBand, upperMzBand);
      curMz += step;
    }
    return c;
  }
  
  /**
   * returns the time value of the peak
   * @param cx the chromatogram where the peak should be found
   * @param scanNumber the scan number to find the peak
   * @return the float value of the peak retention time
   */
  public float getPeak(CgChromatogram cx, int scanNumber, int quantMethod)
  {
    int index = this.findBaseScanIndex(scanNumber);
    cx.GetMaximumAndAverage();
    cx.FindPeak(index, 1, quantMethod);
    return cx.Value[cx.Peak][0];
  }

  /**
   * reads out the chromatograms with the in process defined input parameters
   * @throws CgException
   */
  protected abstract void readTheChromatograms() throws CgException;
  
  /**
   * reads out one chromatogram
   * @param mz the main mz of the chromatogram
   * @param lowerMzBand the lower threshold
   * @param upperMzBand the upper threshold
   * @return the chromatogram
   * @throws CgException
   */
  protected abstract CgChromatogram readAChromatogram(float mz, float lowerMzBand, float upperMzBand) throws CgException;
  
  protected abstract CgChromatogram readAChromatogram(float mz, float lowerMzBand, float upperMzBand,float smoothRange,int smoothRepeats) throws CgException;
  
  protected abstract CgChromatogram readAChromatogram(float mz, float lowerMzBand, float upperMzBand,float smoothRange,int smoothRepeats, float startTime, float stopTime) throws CgException;
  
  /**
   * finds the index in the chromatogram array (only one type of spectra: MS or MS/MS) 
   * @param scanNumber the orifinal scan number (both types like in the raw data)
   * @return the the index in the chromatogram array
   */
  protected abstract int findBaseScanIndex(int scanNumber);
  
  /**
   * Returns the set parameters
   * @return the parameters
   */
  public CgParameterSet getM_params()
  {
    return m_params;
  }

  public abstract int getTimeShiftedScanNumber(int scanNumber, float timeShift);
  
  public static CgProbe calculateOneArea(CgChromatogram chrom,int scan, int mode){
    chrom.FindPeak(scan, 1, mode);
    if (chrom.anythingThere){
      chrom.GetBackground(50);
      chrom.GetAreaAndTime();
    }
    CgProbe probe = new CgProbe(0,1);
    probe = Analyzer.copyResults(probe, chrom);
    return probe;
  }
  

}
