/*
 * Created on 24.06.2005
 * Erik Koerner, FH Joanneum
 * erik.koerner@fh-joanneum.at
 */
package at.tugraz.genome.maspectras.quantification;

import java.io.Serializable;

/**
 * Represents a Level 2 or 3 scan identifying a Peptide. Used as input for the analysis
 * process, and holds the area and range as results.
 * 
 * @author Erik Koerner
 * @version 25.06.2005
 */
public class CgProbe implements Serializable
{
    public int 			Scan;			    // Scan Number (in mzXml File)
    public int 			Charge;			    // Charge 
    
    public float        LowerMzBand;        // Lower m/z area
    public float        Mz;                 // m/z Value;
    public float        UpperMzBand;        // Upper m/z area
    
    public float		LowerValley;        // The Lower Valley Retention Time
    public float		Peak;			    // The Peak Retention Time
    public float        PeakAmplitudeRaw;   // The Peak's Amplitude
    public float        PeakAmplitudeFit;   // The Peak's Amplitude
    public float		UpperValley;	    // The Upper Valley Retention Time
    
    public float		Background;		    // Background Amplitude
    public float		Area;			    // The calculated Area
    public float		AreaError;		    // It's error
    public CgAreaStatus AreaStatus;		    // 0 ... Flche ist ok
    private Long peakId;
    private Long uniqueIdentifier;
    public int isotopeNumber;
    public boolean greedyProbe;
    public boolean greedyFragment;
    
//    private int loSteepnessPoint_;
//    private int upSteepnessPoint_;
    private boolean lowOverlap;
    private boolean upOverlap;
    
    private float highestIntensity_;
    
    /** chemical formula of found analyte - mandatory for LDA MSn identifications*/
    private String formula_;
    /** MSn level of identified analyte/fragment - mandatory for LDA MSn identifications*/
    private int msLevel_;
    
    private boolean isFromOtherSpecies_;
    
    private Float ApexIntensity;
    private Float LowerValley10;
    private Float LowerValley50;
    private Float UpperValley10;
    private Float UpperValley50;
    
    public CgProbe(int scan, int charge)
    {
        Scan             = scan;
        Charge           = charge;
        
        LowerMzBand      = 0;
        Mz               = 0;
        UpperMzBand      = 0;
        
        LowerValley      = 0;
        Peak             = 0;
        PeakAmplitudeRaw = 0;
        PeakAmplitudeFit = 0;
        UpperValley      = 0;
        
        Background       = 0;
        Area             = 0;
        AreaError        = 0;
        AreaStatus       = CgAreaStatus.Undefined;
        isotopeNumber    = 0;
        greedyProbe = false;
        greedyFragment = false;
        this.lowOverlap = false;
        this.upOverlap = false;
        highestIntensity_ = 0f;
        this.msLevel_ = 1;
        this.formula_ = null;
        this.isFromOtherSpecies_ = false;
        
        LowerValley10 = null;
        LowerValley50 = null;
        UpperValley10 = null;
        UpperValley50 = null;

//        loSteepnessPoint_ = -1;
//        upSteepnessPoint_ = -1;
    }
    
    /**
     * constructor for LDA MSn identifications
     * @param scan scan number - obsolete
     * @param charge charge state of analyte
     * @param msLevel MSn level of identification
     * @param formula chemical formula of fragment/analyte
     */
    public CgProbe (int scan, int charge, int msLevel, String formula){
      this(scan,charge);
      msLevel_ = msLevel;
      formula_ = formula;
    }
    
    public CgProbe (CgProbe probe){
      this(probe.Scan,probe.Charge,probe.msLevel_,probe.formula_);
      LowerMzBand      = probe.LowerMzBand;
      Mz               = probe.Mz;
      UpperMzBand      = probe.UpperMzBand;
      
      LowerValley      = probe.LowerValley;
      Peak             = probe.Peak;
      PeakAmplitudeRaw = probe.PeakAmplitudeRaw;
      PeakAmplitudeFit = probe.PeakAmplitudeFit;
      UpperValley      = probe.UpperValley;
      
      Background       = probe.Background;
      Area             = probe.Area;
      AreaError        = probe.AreaError;
      AreaStatus       = probe.AreaStatus;
      isotopeNumber    = probe.isotopeNumber;
      greedyProbe = probe.greedyProbe;
      greedyFragment = probe.greedyFragment;
      this.lowOverlap = probe.lowOverlap;
      this.upOverlap = probe.upOverlap;
      this.msLevel_ = probe.msLevel_;
      this.formula_ = probe.formula_;
      this.isFromOtherSpecies_ = probe.isFromOtherSpecies_;
      this.ApexIntensity = probe.ApexIntensity;
      this.LowerValley10 = probe.LowerValley10;
      this.LowerValley50 = probe.LowerValley50;
      this.UpperValley10 = probe.UpperValley10;
      this.UpperValley50 = probe.UpperValley50;
    }
    
    /**
     * Determines whether another CgProbe covers the same time range.
     * 
     * @param p
     *      CgProbe to be checked
     * @return
     *      true if the range is equal
     */
    public boolean EqualValleys(CgProbe p)
    {
        if (p.LowerValley!=LowerValley) return false;
        if (p.UpperValley!=UpperValley) return false;
        if (p.Charge!=Charge) return false;
        return true;
    }
    
    public boolean isCoveredByThisProbe(CgProbe p){
      if (p.Charge!=Charge)
        return false;
      if (p.LowerValley<=LowerValley&&p.UpperValley>=UpperValley)
        return true;
      else
        return false;
    }

    public Long getPeakId()
    {
      return peakId;
    }

    public void setPeakId(Long peakId)
    {
      this.peakId = peakId;
    }

    public Long getUniqueIdentifier()
    {
      return uniqueIdentifier;
    }

    public void setUniqueIdentifier(Long uniqueIdentifier)
    {
      this.uniqueIdentifier = uniqueIdentifier;
    }
    
    public boolean hasLoOverlap(){
      return this.lowOverlap;
//      if (this.loSteepnessPoint_!=-1)
//        return true;
//      else
//        return false;
    }
    
    public void setLoOverlap(boolean loOverlap){
      this.lowOverlap = loOverlap;
    }
    
    public boolean hasUpOverlap(){
      return this.upOverlap;
//      if (this.upSteepnessPoint_!=-1)
//        return true;
//      else
//        return false;
    }
    
    public void setUpOverlap(boolean upOverlap){
      this.upOverlap = upOverlap;
    }
    
    public String toString(){
      String probeString = this.Area+";"+this.Scan+";"+this.Charge+";"+this.LowerMzBand+";"+this.Mz+";"+this.UpperMzBand+";";
      probeString +=  this.LowerValley+";"+this.Peak+";"+this.UpperValley+this.AreaStatus+";"+this.peakId+";"+this.uniqueIdentifier+";";
      probeString +=  this.isotopeNumber+";"+this.greedyProbe+";"+this.greedyFragment+";"+this.lowOverlap+";"+this.upOverlap+";"+this.AreaStatus;
      return probeString;
    }

    public float getHighestIntensity()
    {
      return highestIntensity_;
    }

    public void setHighestIntensity(float highestIntensity)
    {
      this.highestIntensity_ = highestIntensity;
    }

    /**
     * 
     * @return chemical formula of identified analyte/fragment
     */
    public String getFormula()
    {
      return formula_;
    }

    /**
     * 
     * @return MS level of the identified analyte/fragment
     */
    public int getMsLevel()
    {
      return msLevel_;
    }

    /**
     * this is for MSn analysis
     * @return true if the fragment belongs to another isobaric species
     */
    public boolean isFromOtherSpecies()
    {
      return isFromOtherSpecies_;
    }

    /**
     * this is for MSn analysis
     * @param isFromOtherSpecies true if the fragment belongs to another isobaric species
     */
    public void setFromOtherSpecies(boolean isFromOtherSpecies)
    {
      this.isFromOtherSpecies_ = isFromOtherSpecies;
    }

    public Float getApexIntensity()
    {
      return ApexIntensity;
    }

    public void setApexIntensity(Float apexIntensity)
    {
      ApexIntensity = apexIntensity;
    }

    public Float getLowerValley10()
    {
      return LowerValley10;
    }

    public void setLowerValley10(Float lowerValley10)
    {
      LowerValley10 = lowerValley10;
    }

    public Float getLowerValley50()
    {
      return LowerValley50;
    }

    public void setLowerValley50(Float lowerValley50)
    {
      LowerValley50 = lowerValley50;
    }

    public Float getUpperValley10()
    {
      return UpperValley10;
    }

    public void setUpperValley10(Float upperValley10)
    {
      UpperValley10 = upperValley10;
    }

    public Float getUpperValley50()
    {
      return UpperValley50;
    }

    public void setUpperValley50(Float upperValley50)
    {
      UpperValley50 = upperValley50;
    }
    
    
}
