package at.tugraz.genome.maspectras.quantification;


import at.tugraz.genome.maspectras.rawfilereader.MSBinaryData;
import at.tugraz.genome.maspectras.rawfilereader.MSBinaryParser;
import at.tugraz.genome.maspectras.rawfilereader.MSBinaryParserException;
import at.tugraz.genome.maspectras.rawfilereader.MSScanInfo;
import at.tugraz.genome.maspectras.rawfilereader.MSRawParserConstants;
import at.tugraz.genome.maspectras.rawfilereader.MSMZDataPoint;
import at.tugraz.genome.maspectras.utils.Calculator;

import java.io.RandomAccessFile;
import java.io.File;
import java.io.FileNotFoundException;
import java.util.StringTokenizer;


/**
 * Adapter to parse the binary XCalibur files
 * This file does not implement that it readsJustParts of the file for memory savings
 * (mxXML and mzData does)
 * @author Juergen Hartler
 *
 */
public class CgXCaliburRawReader implements CgReader
{
  /** interface to add scans */
  private CgIAddScan            adder;
  /** start time of the scans */
  private float                 startTime;
  /** stop time of the scans */
  private float                 stopTime;
  /** number of scans */
  private int                   scanCount;
  /** last parsed scan*/
  private CgScan                lastScan;
  
  private int lowestMz = 1000000 * CgDefines.mzMultiplicationFactorForInt;

  private int highestMz = 0;
  
  private float lowerThreshold = 0;
  
  private float upperThreshold = 1000000;

  private int multiplicationFactorForInt_;
  
  /**
   * 
   * @param callbacks interface to add scans
   */
  public CgXCaliburRawReader(CgIAddScan callbacks)
  {
      adder = callbacks;
      multiplicationFactorForInt_ = CgDefines.mzMultiplicationFactorForInt;
      lowestMz = 1000000 * CgDefines.mzMultiplicationFactorForInt;
  }
  
  public CgXCaliburRawReader(CgIAddScan callbacks, int multiplicationFactorForInt)
  {
      this(callbacks);
      multiplicationFactorForInt_ = multiplicationFactorForInt;
      lowestMz = 1000000 * multiplicationFactorForInt_;
  }
  /**
   * Reads in the XCalibure RAW file
   * 
   * @param fileName
   *          absolute path to the file
   */
  public void ReadFile(String fileName) throws CgException
  {
    try{
      MSBinaryParser binaryParser = new MSBinaryParser();
      RandomAccessFile raf=new RandomAccessFile(new File(fileName), "r");
      MSBinaryData msdata = binaryParser.parseFile(raf);
      raf.close();
      MSScanInfo scanlist[]= msdata.getScans();
      msdata=null;
      CgScan    sc                = null;
      
      
      CgScanHeader myHeader = new CgScanHeader();
      myHeader.ScanCount = scanlist.length;
      myHeader.StartTime = ((Double)scanlist[0].getScanAttribute(MSRawParserConstants.ATTRIBUTE_RETENTION_TIME)).floatValue();
      myHeader.EndTime = ((Double)scanlist[scanlist.length-1].getScanAttribute(MSRawParserConstants.ATTRIBUTE_RETENTION_TIME)).floatValue();
      adder.AddHeader(myHeader);
      for (int i=0; i!=scanlist.length;i++){
        MSScanInfo scanInfo = scanlist[i];
        if (((Integer)scanInfo.getScanAttribute(MSRawParserConstants.ATTRIBUTE_MS_LEVEL)).intValue()==1){
          sc = new CgScan(scanInfo.getHitCount());
          sc.Num = (i+1);
          sc.MsLevel = ((Integer)scanInfo.getScanAttribute(MSRawParserConstants.ATTRIBUTE_MS_LEVEL)).intValue();
          sc.RetentionTime = ((Double)scanInfo.getScanAttribute(MSRawParserConstants.ATTRIBUTE_RETENTION_TIME)).floatValue();
          sc.LowMz = ((Double)scanInfo.getScanAttribute(MSRawParserConstants.ATTRIBUTE_SCAN_START_MZ)).floatValue();
          if (sc.LowMz<this.lowestMz) this.lowestMz = Math.round(sc.LowMz*multiplicationFactorForInt_);
          sc.HighMz = ((Double)scanInfo.getScanAttribute(MSRawParserConstants.ATTRIBUTE_SCAN_END_MZ)).floatValue();
          if (sc.HighMz>this.highestMz) this.highestMz = Math.round(sc.HighMz*multiplicationFactorForInt_);
          sc.BasePeakMz = ((Double)scanInfo.getScanAttribute(MSRawParserConstants.ATTRIBUTE_BASE_PEAK_MZ)).floatValue();
          sc.BasePeakIntensity = ((Double)scanInfo.getScanAttribute(MSRawParserConstants.ATTRIBUTE_BASE_PEAK_INTENSITY)).floatValue();
          sc.TotIonCurrent = ((Double)scanInfo.getScanAttribute(MSRawParserConstants.ATTRIBUTE_TOTAL_ION_CURRENT)).floatValue();
          MSMZDataPoint[] hitlist = scanInfo.getHitList();
          for (int j=0; j!=hitlist.length;j++){
            sc.Scan[j][0] = (new Double(hitlist[j].mass)).floatValue();
//          System.out.println(j);
            sc.Scan[j][1] = (new Double(hitlist[j].detectorhitcount)).floatValue();
          }
          adder.AddScan(sc);
          this.lastScan = sc;
        }else{
          if (lastScan!=null) lastScan.AddSubscanNumber(i+1);
          else              throw new CgException("No base scan for subscan.");                      
        }
      }
      System.out.println("The parsing is done successfully");
      System.out.println(scanlist.length);
    }catch(FileNotFoundException fnx){
      fnx.printStackTrace();
      throw new CgException(fnx.getMessage());
    }catch(MSBinaryParserException mpx){
      mpx.printStackTrace();
      throw new CgException(mpx.getMessage());
    }catch(Exception ex){
      ex.printStackTrace();
    }
  }
  
  public void ReadFile(String fileName, boolean readJustMzMaxima) throws CgException{
    this.ReadFile(fileName);
  }
  
  public int getHighestMz()
  {
    return this.highestMz;
  }

  public int getLowestMz()
  {
    return this.lowestMz;
  }

  public void setLowerThreshold(float lowerThreshold)
  {
    this.lowerThreshold = lowerThreshold;
  }

  public void setUpperThreshold(float upperThreshold)
  {
    this.upperThreshold = upperThreshold;
  }

  @Override
  public boolean usesPolaritySwitching()
  {
    // TODO Auto-generated method stub
    return false;
  }

}
