package at.tugraz.genome.maspectras.quantification;


import java.util.Hashtable;
import java.util.Vector;
import java.util.StringTokenizer;


/*import org.jfree.chart.ChartFactory;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.plot.PlotOrientation;
import org.jfree.data.xy.XYDataset;
import org.jfree.data.xy.XYSeries;
import org.jfree.data.xy.XYSeriesCollection;*/


/**
 * Analyzer to calculate the peak area of one peptide; the used file type is the chromatogram file
 * @author Juergen Hartler
 *
 */
public class ChromaAnalyzer extends Analyzer
{
  
  /** the file reader*/
  protected ChromatogramReader reader_;
  
  
  /**
   * This constructor is used on the cluster; all parameters are definable
   * @param headerFilePath path to the header file
   * @param indexFilePath path to the index file
   * @param retentionTimeFilePath path to the retention-time file
   * @param chromatogramFilePath path to the chromatogram file
   * @param sparseData for sparse data an interpolation is required
   * @param chromSmoothRange if there is interpolation - the amount of points depend on the smooth range
   * @throws CgException
   */
  public ChromaAnalyzer(String headerFilePath, String indexFilePath, String retentionTimeFilePath, String chromatogramFilePath, boolean sparseData, float chromSmoothRange)throws CgException{
    super();
    reader_ = new ChromatogramReader(headerFilePath,indexFilePath,retentionTimeFilePath,chromatogramFilePath,sparseData,chromSmoothRange);
    m_chroma = new CgChromatogram[CgDefines.MaxCharge];
  }
  
  /**
   * This constructor takes only the file path of the header file; the rest of the information is read from the header file
   * @param filePath path to the header file
   * @throws CgException
   */
  public ChromaAnalyzer(String filePath) throws CgException{
    this (filePath, false);
  }

  /**
   * This constructor takes only the file path of the header file; the rest of the information is read from the header file
   * @param filePath filePath path to the header file
   * @param useCache should the whole chromatogram file be read into the cache; is not advisable with big files
   * @throws CgException
   */
  public ChromaAnalyzer(String filePath, boolean useCache) throws CgException{
    super();
    reader_ = new ChromatogramReader(filePath,useCache);
    m_chroma = new CgChromatogram[CgDefines.MaxCharge];
  }
  
  /**
   * This constructor may only be used if there is no need to to init the reader
   * and the chromatogram, and it will be done by the calling constructor
   */
  protected ChromaAnalyzer(){
    super();
  }
  
  
  protected int findBaseScanIndex(int scanNumber)
  {
    return reader_.findBaseScanIndex(scanNumber);
  }

  protected void readTheChromatograms()throws CgException
  {
    for (int i = 0; i < CgDefines.MaxCharge; i++) {
      if (m_params.Mz[i] > 0) {
        m_chroma[i] = this.readAChromatogram(m_params.Mz[i],m_params.LowerMzBand,m_params.UpperMzBand);
      } else {
        m_chroma[i] = null;
      }
    }  
  }
  
  protected CgChromatogram readAChromatogram(float mz, float lowerMzBand, float upperMzBand) throws CgException{
    return this.readAChromatogram(mz, lowerMzBand, upperMzBand, 0.5f, 10);
  }
  
  protected CgChromatogram readAChromatogram(float mz, float lowerMzBand,float upperMzBand,float smoothRange,int smoothRepeats) throws CgException{
    return this.readAChromatogram(mz, lowerMzBand, upperMzBand, 1, smoothRange, smoothRepeats);
  }
  
  protected CgChromatogram readAChromatogram(float mz, float lowerMzBand,float upperMzBand, int msLevel, float smoothRange,int smoothRepeats) throws CgException{
    return this.readAChromatogram(mz, lowerMzBand, upperMzBand, msLevel, smoothRange, smoothRepeats, -1f, Float.MAX_VALUE);
  }
  
  protected CgChromatogram readAChromatogram(float mz, float lowerMzBand, float upperMzBand,float smoothRange, int smoothRepeats, float startTime, float stopTime) throws CgException{
    return readAChromatogram(mz, lowerMzBand, upperMzBand, 1,smoothRange,  smoothRepeats, startTime, stopTime);
  }

  protected CgChromatogram readAChromatogram(float mz, float lowerMzBand, float upperMzBand,int msLevel,float smoothRange, int smoothRepeats,  float startTime, float stopTime) throws CgException{
    return readAChromatogram(mz, lowerMzBand, upperMzBand, msLevel,smoothRange,  smoothRepeats, smoothRange, 0, startTime, stopTime);
  }
  
  protected CgChromatogram readAChromatogram(float mz, float lowerMzBand, float upperMzBand,int msLevel,float smoothRange, int smoothRepeats, float meanSmoothRange, int meanSmoothRepeats, float startTime, float stopTime) throws CgException{
//  long time = System.currentTimeMillis();
    CgChromatogram cx = reader_.readChromatogram(mz - lowerMzBand,mz + upperMzBand,msLevel);
    cx.Mz = mz;
    cx.LowerMzBand = lowerMzBand;
    cx.UpperMzBand = upperMzBand;
    if (startTime!=-1f || stopTime!=Float.MAX_VALUE){
      boolean foundStart = false;
      boolean foundStop = false;
      for (int i=0;i!=cx.ScanCount;i++){
        if (!foundStart && startTime<cx.Value[i][0]){
          cx.startSmoothScan_ = i-1;
          foundStart = true;
        }
        if (!foundStop && stopTime<cx.Value[i][0]){
          cx.stopSmoothScan_ = i+1;
          foundStop = true;
        }
      }
    }
    boolean copyRawData = true;
    if (meanSmoothRange>0){
      cx.smoothMean(meanSmoothRange, meanSmoothRepeats,copyRawData);
      copyRawData = false;
    }
//    System.out.println("Reading Time: "+(System.currentTimeMillis()-time));
//    time = System.currentTimeMillis();
    cx.Smooth(smoothRange, smoothRepeats, copyRawData);
    
//    System.out.println("Smoothing Time: "+(System.currentTimeMillis()-time));
    return cx;
  }
  
  /**
   * reads a chromatogram that contains only the contents of a 3D probe
   * @param probe the 3D probe
   * @param msLevel the MS-level
   * @return a chromatogram that contains only the contents of a 3D probe
   * @throws CgException something wrong with the reading
   */
  public CgChromatogram readJustIntensitiesOfInterest(Probe3D probe, int msLevel) throws CgException{
    CgChromatogram cx = reader_.readJustIntensitiesOfInterest(probe, msLevel);
    return cx;
  }
  
  /**
   * reads a chromatogram that contains only the contents of a certain m/z and time range - used for MS/MS peak quantitation
   * @param startMz the start m/z value of the chromatogram
   * @param stopMz the stop m/z value of the chromatogram
   * @param startTime the start retention time of the chromatogram - use -1f for neglecting a start time
   * @param stopTime the stop retention time of the chromatogram - use -1f for neglecting a stop time
   * @param msLevel the MS-level of the crhomatogram
   * @return a chromatogram that contains only the contents of a certain m/z and time range - used for MS/MS peak quantitation
   * @throws CgException something wrong with the reading
   */
  public CgChromatogram readJustIntensitiesOfInterest(float startMz, float stopMz, float startTime, float stopTime, int msLevel) throws CgException{
    CgChromatogram cx = reader_.readJustIntensitiesOfInterest(startMz, stopMz, startTime, stopTime, msLevel);
    return cx;
  }

  
  /** reads an m/z profile from the chrom file and smooths it*/
  protected Vector<CgChromatogram> readProfiles(Vector<CgProbe> probes, float mzTolerance, float timeTolerance, float maxTimeDeviation,
      float mzSmoothRange, int smoothRepeats, float meanSmoothRange, int meanSmoothRepeats, int msLevel) throws CgException{
    Vector<CgChromatogram> chroms =  reader_.readProfiles(probes, mzTolerance, timeTolerance, maxTimeDeviation,0,0, msLevel);
    for (CgChromatogram chrom : chroms){
      boolean copyRawData = true;
      if (meanSmoothRange>0){
        chrom.smoothMean(meanSmoothRange, meanSmoothRepeats,copyRawData);
        copyRawData = false;
      }
//      System.out.println("Reading Time: "+(System.currentTimeMillis()-time));
//      time = System.currentTimeMillis();
      chrom.Smooth(mzSmoothRange, smoothRepeats, copyRawData);
    }
    return chroms;
/****    if (probes.size()>0){
      float mz = probes.get(0).Mz;
      System.out.println(mz);
      CgChromatogram cx = reader_.readChromatogram(mz - lowerMzBand,mz + upperMzBand);
      cx.Mz = mz;
      cx.LowerMzBand = lowerMzBand;
      cx.UpperMzBand = upperMzBand;
      cx.Smooth(0.5f, smoothRepeats);
    }
    return null;//    return cx;*/
  }
  
  public  String[] getRawLines (float mz, float lowerMzBand, float upperMzBand, int msLevel) throws CgException{
    String[] rawLines = reader_.getRawLines(mz - lowerMzBand,mz + upperMzBand, msLevel);
    int resolutionFactor = this.getResolutionFactor();
    if (rawLines.length<1){
      rawLines = new String[Integer.valueOf((new StringTokenizer(String.valueOf((upperMzBand+lowerMzBand)*resolutionFactor),".")).nextToken())];
      for (int i=0;i!=rawLines.length;i++){
        rawLines[i] = "";
      }
      return rawLines;
    }
    if ((mz - lowerMzBand)*resolutionFactor<reader_.getLowestMz_()){
      String[] newRawLines = new String[Integer.valueOf((new StringTokenizer(String.valueOf((upperMzBand+lowerMzBand)*resolutionFactor),".")).nextToken())];
      for (int i=0;i!=newRawLines.length;i++){
        if (i<(newRawLines.length-rawLines.length))
          newRawLines[i] = "";
        else
          newRawLines[i] = rawLines[i-newRawLines.length+rawLines.length];
      }
      return newRawLines;
    }
    if ((mz + upperMzBand)*resolutionFactor>reader_.getHighestMz_()){
      String[] newRawLines = new String[Integer.valueOf((new StringTokenizer(String.valueOf((upperMzBand+lowerMzBand)*resolutionFactor),".")).nextToken())];
      for (int i=0;i!=newRawLines.length;i++){
        if (i<rawLines.length)
          newRawLines[i] = rawLines[i];
        else
          newRawLines[i] = "";
      }
      return newRawLines;
    }  
    return rawLines;
  }

  public Hashtable<Integer,Float> getRetentionTimes()
  {
    return reader_.getRetentionTimes();
  }

  public Hashtable<Integer,Float> getRetentionTimes(int msLevel)
  {
    return reader_.getRetentionTimes(msLevel);
  }

  
  public int getNumberOfScans(){
    return reader_.getNumberOfScans();
  }
  
  public int getResolutionFactor(){
    return reader_.getMultiplicationFactorForInt_()/reader_.getLowestResolution_();
  }
  
  public int getTimeShiftedScanNumber(int scanNumber, float timeShift){
    int shiftedScanNumber =  scanNumber;
    Hashtable<Integer,Float> retentionTimes = this.getRetentionTimes();
    Vector<Integer> originalScanNumbers = reader_.getOriginalScanNumbers();
    int baseScanIndex = this.findBaseScanIndex(scanNumber);
    float timeToLookAt = retentionTimes.get(baseScanIndex)+timeShift;
    if (timeShift>0){
      int lastScan = baseScanIndex;
      for (int i=baseScanIndex;i!=retentionTimes.size();i++){
        Integer scan = new Integer(i);
        if (retentionTimes.containsKey(scan)&&retentionTimes.get(scan).floatValue()>timeToLookAt){
          return originalScanNumbers.get(lastScan).intValue();
        }else{
          lastScan = scan.intValue();
        }
      }
    }else if (timeShift<0){
      for (int i=baseScanIndex;i!=-1;i--){
        Integer scan = new Integer(i);
        if (retentionTimes.containsKey(scan)&&retentionTimes.get(scan).floatValue()<timeToLookAt)
          return originalScanNumbers.get(i).intValue();
      }
    }
    return shiftedScanNumber;
  }
  
  /**
   * caches the spectra of a certain precursor mass
   * @param from lower m/z threshold for precursor m/z
   * @param to upper m/z threshold for precursor m/z
   * @param minIntsForNoiseRemoval minimum amount of detected signals to start a noise removal
   * @return
   * @throws CgException
   */
  public Hashtable<Integer,Boolean> prepareMSnSpectraCache(float from, float to, int minIntsForNoiseRemoval) throws CgException{
    return reader_.prepareMSnSpectraCache(from, to, minIntsForNoiseRemoval);
  }
  
  /**
   * 
   * @return the cached spectra for a certain precursor range
   */
  public Hashtable<Integer,Hashtable<Integer,String>> getMSnSpectraCache(){
    return reader_.getMSnSpectraCache();
  }
  
  /**
   * 
   * @return the cached noise for a certain precursor range
   */
  public Hashtable<Integer,Hashtable<Integer,Float>> getMSnSpectraNoise(){
    return reader_.getMSnSpectraNoise();
  }
  
  /**
   * ATTENTION: prepareMSnSpectraCache has to be called before
   * queries the data if there are spectra in a certain time range
   * @param startTime lower time threshold in seconds
   * @param stopTime upper time threshold in seconds
   * @param msLevel MSn level on which the check shall be performed
   * @return true if there is any data at this time range
   * @throws CgException
   */
  public boolean areMSnSpectraInThisRegion(float startTime, float stopTime, int msLevel) throws CgException{
    return reader_.areMSnSpectraInThisRegion(startTime, stopTime, msLevel);
  }

  /**
   * ATTENTION: countMSnSpectraOfRegion has to be called before
   * queries the data if there are spectra in a certain time range
   * @param startTime lower time threshold in seconds
   * @param stopTime upper time threshold in seconds
   * @param msLevel MSn level on which the check shall be performed
   * @return the amount of MSn spectra in a certain time range
   * @throws CgException
  */
  public int countMSnSpectraOfRegion(float startTime, float stopTime, int msLevel) throws CgException{
    return reader_.countMSnSpectraOfRegion(startTime, stopTime, msLevel);
  }

  /**
   * ATTENTION: prepareMSnSpectraCache has to be called before
   * extracts intensity information
   * @param levels for which MSn levels should the base peak values be extracted
   * @param probes the probes specifying the time range where the base peak value shall be calculated
   * @param mzTolerance +/- MSn range for the base peak
   * @return hash table containing as key the MSn level and as value the base peak intensity
   * @throws CgException
   */
  public Hashtable<Integer,Float> extractBasePeakValues(Vector<Integer> levels, Vector<CgProbe> probes, float mzTolerance) throws CgException {
    float[] startStopTime = getStartStopTimeFromProbes(probes);
    return  extractBasePeakValues(levels, startStopTime[0], startStopTime[1], mzTolerance);  
  }
  
  /**
   * ATTENTION: prepareMSnSpectraCache has to be called before
   * extracts intensity information
   * @param levels for which MSn levels should the base peak values be extracted
   * @param startTime lower time threshold in seconds
   * @param stopTime upper time threshold in seconds
   * @param mzTolerance +/- MSn range for the base peak
   * @return hash table containing as key the MSn level and as value the base peak intensity
   * @throws CgException
   */
  public Hashtable<Integer,Float> extractBasePeakValues(Vector<Integer> levels, float startTime, float stopTime, float mzTolerance) throws CgException {
    return reader_.extractBasePeakValues(levels, startTime, stopTime, mzTolerance);
  }
  
  /**
   * ATTENTION: prepareMSnSpectraCache has to be called before
   * calculates how many percents of the used spectra are covered by found hits
   * @param ms1Probes probes the probes specifying the time range where the spectrum coverage shall be calculated
   * @param foundHits identified m/z values
   * @param notCounted hits that originate from another species; these hits should be excluded from the spectrum coverage
   * @param levels for which MSn levels should the base peak values be extracted
   * @param basePeakCutoff relational value for discarding small peak intensities
   * @return hash table containing the coverages
   * @throws CgException
   */
  public Hashtable<Integer,Float> calculateSpectrumCoverage(Vector<CgProbe> ms1Probes, Vector<CgProbe> foundHits, Vector<CgProbe> notCounted, Vector<Integer> levels, float basePeakCutoff) throws CgException {
    float[] startStopTime = getStartStopTimeFromProbes(ms1Probes);
    return reader_.calculateSpectrumCoverage(foundHits, notCounted, levels, startStopTime[0], startStopTime[1],basePeakCutoff);
  }
  
  /**
   * ATTENTION: prepareMSnSpectraCache has to be called before
   * calculates how many percents of the used spectra are covered by found hits
   * @param foundHits identified m/z values
   * @param notCounted hits that originate from another species; these hits should be excluded from the spectrum coverage
   * @param levels for which MSn levels should the base peak values be extracted
   * @param startTime lower time threshold in seconds
   * @param stopTime upper time threshold in seconds
   * @param basePeakCutoff relational value for discarding small peak intensities
   * @return hash table containing the coverages
   * @throws CgException
   */
  public Hashtable<Integer,Float> calculateSpectrumCoverage(Vector<CgProbe> foundHits, Vector<CgProbe> notCounted, Vector<Integer> levels, float startTime, float stopTime, float basePeakCutoff) throws CgException {
    return reader_.calculateSpectrumCoverage(foundHits, notCounted, levels, startTime, stopTime, basePeakCutoff);
  }

  /**
   * extracts the lower and upper time threshold that are stored in CgProbe.LowerValley and CgProbe.UpperValley
   * @param probes the probes specifying the time range
   * @return float[0]: lower time threshold; float[1] upper time threshold
   */
  public float[] getStartStopTimeFromProbes(Vector<CgProbe> probes){
    float lowestRt = Float.MAX_VALUE;
    float highestRt = 0;
    for (CgProbe probe : probes){
      if (probe.LowerValley < lowestRt) lowestRt = probe.LowerValley;
      if (probe.UpperValley > highestRt) highestRt = probe.UpperValley;
    }
    float[] startStopTime = new float[2];
    startStopTime[0] = lowestRt;
    startStopTime[1] = highestRt;
    return startStopTime;
  }
}
