package at.tugraz.genome.maspectras.quantification;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.DataInputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.Vector;
import java.math.BigDecimal;
import java.nio.ByteBuffer;
import java.nio.FloatBuffer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Hashtable;

import at.tugraz.genome.dbutilities.Base64;
import at.tugraz.genome.maspectras.utils.Calculator;
import at.tugraz.genome.util.FloatMatrix;
import at.tugraz.genome.util.IndexedLineNumberReader;
import at.tugraz.genome.util.index.IndexFileException;

/**
 * Reads the chromatograms from the chrom files
 * This reader makes use of an indexed file to offer efficient file access
 * @author Juergen Hartler
 *
 */
public class ChromatogramReader extends IndexedLineNumberReader
{
  public static final String CHROMATOGRAM_HEADER_FILE_MSMS_TYPE_FULL = "fullMSMSScan";
  
  public static final String CHROMATOGRAM_HEADER_FILE_MSMS_TYPE_PRECURSOR = "precursorFragmentation";
  
  /** warning message when there are no MS/MS spectra present*/
  private static final String NO_MSMS_SPECTRA = "There are no MS/MS spectra for this precursor!";

  
  /** the lowest possible m/z value*/
  private int lowestMz_;
  /** the highest possible m/z value*/
  private int highestMz_;
  /** the multiplication factor to get integer line number from the m/z value*/
  private int multiplicationFactorForInt_;
  /** the lowest resolution value*/
  private int lowestResolution_;
  /** the amount of scans (just one type: MS or MS/MS)*/
  private int numberOfScans;
  /** lookup from the just one type (MS) of scans to the original ones*/ 
  private Vector<Integer> originalScanNumbers = new Vector<Integer>();
  /** lookup from the just one type (MS or MS/MS) of scans to the retention times*/ 
  private Hashtable<Integer,Float> retentionTimes_;
  /** the retention as obtained from the chrom file*/
  private Hashtable<Integer,Float> retentionTimesOriginal_;
  /** the chromatogram file in the cache (if desired)*/
  private String[] cachedChromatogramFile;
  /** should the chromatogram file be put into the cache*/
  private boolean chromCache_;
  /** the path to the retention-time file*/
  private String retentionFile_;
  /** the highest MS-Level*/
  private int highestMsLevel_;
  /** the MS/MS-Type*/
  private String msmsType_;
  /** MS/MS index files*/
  private Vector<String> msmsIndexFiles_;
  /** MS/MS chrom files*/
  private Vector<String> msmsChromFiles_;
  /** MS/MS ret files*/
  private Vector<String> msmsRetFiles_;
  /** the MS/MS retention times*/
  private Hashtable<Integer,Hashtable<Integer,Float>> msmsRetentionTimes_;
  /** lookup from scan numbers (all MS levels) to retention time*/
  private Hashtable<Integer,Float> msmsScanNrToRetentionTime_;
  /** lookup from the just one type (MS) of scans to the original ones*/ 
  private Hashtable<Integer,Vector<Integer>> msmsOriginalScanNumbers_;
  private Hashtable<Integer,Hashtable<Integer,Integer>> fromScanToConsecutiveNumber_;
  protected Hashtable<Integer,Integer> msmsNrOfScans_ = new Hashtable<Integer,Integer>();
  private Hashtable<Integer,Hashtable<Integer,Integer>> cachedMsMsLines_;
  private Hashtable<Integer,Hashtable<Integer,Long>> cachedMsMsIndices_;
  private Hashtable<Integer,Integer> msmsNrOfLines_;
  
  private int lastReadMsLevel_ = 1;
  private int[] ms2ExtremeValues_ = new int[4];
  
  /** if the data contains sparse measurements - then, interpolation is required*/
  private boolean sparseData_;
  /** if there is interpolation - the amount of points depend on the smooth range*/
  private float chromSmoothRange_;
  
  
  /** hash table containing the cached spectra; first key is the MSn level; second key is the consecutive number in the chrom file*/
  private Hashtable<Integer,Hashtable<Integer,String>> msnSpectraCache_;
  /** hash table containing the estimated noise level for each spectrum; first key is the MSn level; second key is the consecutive number in the chrom file*/
  private Hashtable<Integer,Hashtable<Integer,Float>> msnSpectraNoise_;
  
  public final static float NOISE_CUTOFF_MULTIPLICATOR = 1.5f;
  
  /**
   * constructor with just the header file; the rest of the information is read from the header file
   * @param fileName the path to the header file
   * @throws CgException
   */
  public ChromatogramReader(String fileName) throws CgException{
    this (fileName,false);
  }
  
  /**
   * This constructor is used on the cluster; here all of the information is definable
   * @param headerFilePath path to the header file
   * @param indexFilePath path to the index file
   * @param retentionTimeFilePath path to the retention time file
   * @param chromatogramFilePath path to the chromatogram file
   * @param sparseData - are there sparse time points in MS1 -> interpolation
   * @param chromSmoothRange if there is interpolation - the amount of points depend on the smooth range
   * @throws CgException
   */
  public ChromatogramReader(String headerFilePath, String indexFilePath, String retentionTimeFilePath, String chromatogramFilePath, boolean sparseData, float chromSmoothRange)throws CgException{
    super(headerFilePath, indexFilePath, chromatogramFilePath, true);
    this.retentionFile_ = retentionTimeFilePath;
    this.sparseData_ = sparseData;
    this.chromSmoothRange_ = chromSmoothRange;
    this.readHeader(this.headerFilePath_);
  }
  
  /**
   * constructor with just the header file and if the file should be put into the cache; the rest of the information is read from the header file
   * @param fileName the header file name
   * @param chromCache should the whole chromatogram be in the cache (not advisable for big files)
   * @throws CgException
   */
  public ChromatogramReader(String fileName, boolean chromCache) throws CgException{
    super(fileName,true);
//    String nameWithoutPostfix =  this.removePostFixIfAny(fileName);
//    this.headerFilePath_ = nameWithoutPostfix+".head";
    this.readHeader(this.headerFilePath_);
    this.chromCache_ = chromCache;
    if (this.chromCache_){
      try {
        this.readWholeChromatogramIntoFile();
       }
      catch (IndexFileException e) {
        throw new CgException(e.getMessage());// TODO Auto-generated catch block
      }
    }
  }
  
  /**
   * reads the header-, index-, and retention time file and initializes all of the parameters
   * @param headerFile path to the header file
   * @throws CgException
   */
  public void readHeader (String headerFile) throws CgException{
    this.headerFilePath_ = headerFile;
    ChromatogramHeaderFileReader headerReader = new ChromatogramHeaderFileReader(this.headerFilePath_);
    lastReadMsLevel_ = 1;
    try{
      headerReader.readFile();
    }catch(IndexFileException ifx){
      throw new CgException(ifx.getMessage());
    }
    highestMsLevel_ = 1;
    msnSpectraCache_ = new Hashtable<Integer,Hashtable<Integer,String>>();
    msnSpectraNoise_ = new Hashtable<Integer,Hashtable<Integer,Float>>();
    this.lowestMz_ = headerReader.getLowestMz();
    this.highestMz_ = headerReader.getHighestMz();   
    this.multiplicationFactorForInt_ = headerReader.getMzMultiplicationFactor();
    this.lowestResolution_ = headerReader.getLowestResolution();
    this.numberOfScans = headerReader.getNumberOfScans();
    this.numberOfLines_ = (highestMz_-lowestMz_)/lowestResolution_;
    msmsIndexFiles_ = new Vector<String>();
    msmsChromFiles_ = new Vector<String>();
    msmsRetFiles_ = new Vector<String>();
    msmsNrOfLines_ = new Hashtable<Integer,Integer>();
    msmsNrOfLines_.put(1, numberOfLines_);
    msmsRetentionTimes_ = new Hashtable<Integer,Hashtable<Integer,Float>>();
    msmsScanNrToRetentionTime_ = new Hashtable<Integer,Float>();
    msmsOriginalScanNumbers_ = new Hashtable<Integer,Vector<Integer>>();
    fromScanToConsecutiveNumber_ = new Hashtable<Integer,Hashtable<Integer,Integer>>();
    msmsNrOfScans_ = new Hashtable<Integer,Integer>();
    cachedMsMsLines_ = new Hashtable<Integer,Hashtable<Integer,Integer>>();
    cachedMsMsIndices_ = new Hashtable<Integer,Hashtable<Integer,Long>>();
    try{
      highestMsLevel_ = headerReader.getMsLevel();
    } catch (IndexFileException idx){}
    try{
      if (highestMsLevel_>1){
        msmsType_ = headerReader.getMsMsType();
        for (int i=2;i<=highestMsLevel_;i++){
          int nrOfMsMsScans = headerReader.getNumberOfScans(i);
          msmsNrOfScans_.put(i, nrOfMsMsScans);
          if (this.indexFilePath_==null || this.indexFilePath_.length()<1)
            msmsIndexFiles_.add(headerReader.getIndexFilePath(i));
          else msmsIndexFiles_.add(this.indexFilePath_+String.valueOf(i));
          if (this.dataFilePath_==null || this.dataFilePath_.length()<1)
            msmsChromFiles_.add(headerReader.getIndexedFilePath(i));
          else msmsChromFiles_.add(this.dataFilePath_+String.valueOf(i));
          String retTimeFile = null;
          if (this.retentionFile_==null||this.retentionFile_.length()<1)
            retTimeFile = headerReader.getRetentionTimeFile(i);
          else retTimeFile =  this.retentionFile_+String.valueOf(i);
          msmsRetFiles_.add(retTimeFile);
          Vector<Integer> originalScanNrs = new Vector<Integer>();
          msmsRetentionTimes_.put(i, readRetentionTimeFile(retTimeFile,originalScanNrs,msmsScanNrToRetentionTime_,nrOfMsMsScans));
          if (msmsType_!=null && msmsType_.equalsIgnoreCase(CHROMATOGRAM_HEADER_FILE_MSMS_TYPE_PRECURSOR)){
            msmsOriginalScanNumbers_.put(i, originalScanNrs);
            Hashtable<Integer,Integer> scanToConsecutive = new Hashtable<Integer,Integer>();
            for (int j=0;j!=originalScanNrs.size();j++){
              scanToConsecutive.put(originalScanNrs.get(j), j);
            }
            fromScanToConsecutiveNumber_.put(i, scanToConsecutive);
          } else if (msmsType_!=null && msmsType_.equalsIgnoreCase(CHROMATOGRAM_HEADER_FILE_MSMS_TYPE_FULL)){
            msmsNrOfLines_.put(i, (headerReader.getHighestMz(i)-headerReader.getLowestMz(i))/lowestResolution_);
          }
        }
      }
      this.numberOfLinesForIndex_ = headerReader.getNumberOfLinesForIndex();
      if (this.indexFilePath_==null || this.indexFilePath_.length()<1)
        this.indexFilePath_ = headerReader.getIndexFilePath();
      if (this.dataFilePath_==null || this.dataFilePath_.length()<1)
        this.dataFilePath_ = headerReader.getIndexedFilePath();
      if (this.retentionFile_==null||this.retentionFile_.length()<1)
        retentionFile_ = headerReader.getRetentionTimeFile();
      msmsIndexFiles_.add(0,indexFilePath_);
      msmsChromFiles_.add(0,dataFilePath_);
      msmsRetFiles_.add(0,retentionFile_);
      retentionTimes_ = readRetentionTimeFile(retentionFile_,originalScanNumbers,null,numberOfScans);
      retentionTimesOriginal_ = retentionTimes_;
      if (sparseData_) interpolateRetentionTimes(chromSmoothRange_);
    }catch (IndexFileException ifx){
      throw new CgException(ifx.getMessage());
    }catch (FileNotFoundException fex){
      throw new CgException(fex.getMessage());
    }catch (IOException iox){
      iox.printStackTrace();
      throw new CgException(iox.getMessage());
    }
  }
  
  /**
   * reads a retention time file and fills the corresponding hash tables
   * @param retFilePath the path to the retention time file
   * @param originalScanNumbers vector to be filled containing the original scan numbers
   * @param msmsScanNrToRetentionTime lookup table for scan number to retention time - this hash table contains all MS-levels 
   * @param amountOfScans the total number of scans
   * @return lookup table; key is a consecutive scan number of this MS-level; value is the retention time
   * @throws IOException
   */
  private Hashtable<Integer,Float> readRetentionTimeFile(String retFilePath, Vector<Integer> originalScanNumbers,
      Hashtable<Integer,Float> msmsScanNrToRetentionTime, int amountOfScans) throws IOException{
    Hashtable<Integer,Float> retTimes = new Hashtable<Integer,Float>();
    DataInputStream stream = new DataInputStream(new BufferedInputStream(new FileInputStream(retFilePath)));
    for (int i=0; i!=amountOfScans;i++){
      int scanNumber = stream.readInt();
      originalScanNumbers.add(scanNumber);
      float rt = stream.readFloat();
      retTimes.put(new Integer(i), rt);
      if (msmsScanNrToRetentionTime!=null) msmsScanNrToRetentionTime.put(scanNumber, rt);
    }
    stream.close();
    return retTimes;
  }
  
  public CgChromatogram readChromatogram(float from, float to, int msLevel) throws CgException{
    return this.readChromatogram(from, to, -1f, -1f, msLevel, null);
  }
  
  /**
   * reads out a chromatogram just of the intensities between the borders of the ellipse or a fixed m/z and/or retention time range
   * @param from start mz the start m/z value of the chromatogram
   * @param to stop mz the stop m/z value of the chromatogram
   * @param startTime the start retention time of the chromatogram - use -1f for neglecting a start time
   * @param stopTime the stop retention time of the chromatogram - use -1f for neglecting a stop time
   * @return a chromatogram that contains only the contents of a certain m/z and time range - used for MS/MS peak quantitation
   * @throws CgException
   */
  protected CgChromatogram readChromatogram(float from, float to, float startTime, float stopTime, int msLevel, Probe3D probe) throws CgException{
    CgChromatogram chrom = null;
//  long startTime = System.currentTimeMillis();
    boolean precursorType = (msLevel>1 && this.msmsType_!=null && this.msmsType_.equalsIgnoreCase(CHROMATOGRAM_HEADER_FILE_MSMS_TYPE_PRECURSOR));
    Hashtable<Integer,Float> rtTimes = getRetentionTimes(msLevel, true);
    int amountOfScans = rtTimes.size();
    chrom = new CgChromatogram(amountOfScans);
    float highestInt = 0f;
    for (int i=0;i!=amountOfScans;i++){
      chrom.Value[i][0] = (rtTimes.get(new Integer(i))).floatValue();
      chrom.Value[i][1] = 0;
    }
    if (precursorType){
      Hashtable<Integer,String> msmsLines = this.msnSpectraCache_.get(msLevel);
      Hashtable<Integer,Float> noiseLevels = msnSpectraNoise_.get(msLevel);
      for (int i=0; i!=rtTimes.size(); i++){
        if (msmsLines==null || !msmsLines.containsKey(i)) continue;
        if (startTime>0 && rtTimes.get(i)<startTime) continue; 
        if (stopTime>0 && rtTimes.get(i)>stopTime) continue; 
        String spectrum = msmsLines.get(i);
        Float noiseThreshold = noiseLevels.get(i)*NOISE_CUTOFF_MULTIPLICATOR;
        spectrum = spectrum.substring(spectrum.indexOf(" ")+1);
        FloatBuffer buffer = ByteBuffer.wrap(Base64.decode(spectrum)).asFloatBuffer();
        int iItems = buffer.limit();
        float mz = 0;
        float intensity = 0;
        for(int iItem = 0; iItem < iItems; iItem++){
          if (iItem%2==0) mz = buffer.get();
          else{
            intensity = buffer.get();
            if (intensity>0 && intensity>noiseThreshold && from<=mz && mz<=to){
              boolean addInt = false;
              if (probe!=null){
                Float[] timeBorders = Calculator.calculateEllipseXBorderValues(mz, probe.getEllipseTimePosition(), probe.getEllipseMzPosition(), 
                  probe.getEllipseTimeStretch(), probe.getEllipseMzStretch());
                if (probe.getLowerHardRtLimit()>=0 && timeBorders[0]<probe.getLowerHardRtLimit())
                  timeBorders[0] = probe.getLowerHardRtLimit();
                if (probe.getUpperHardRtLimit()>=0 && timeBorders[1]>probe.getUpperHardRtLimit())
                  timeBorders[1] = probe.getUpperHardRtLimit();
                if (timeBorders[0]<=chrom.Value[i][0]&&chrom.Value[i][0]<=timeBorders[1]){
                  chrom.Value[i][1] += intensity;
                  addInt = true;
                }  
              }else addInt = true;
              if (addInt){
                chrom.Value[i][1] += intensity;
                if (intensity>highestInt) highestInt = intensity;;
              }
            }
          }
        }
      }
    }else{
      String[] readLines = this.getRawLines(from, to, msLevel);
//    System.out.println(((System.currentTimeMillis()-startTime))+" msecs");
    
      float mzAdduct = ((float)lowestResolution_/(float)multiplicationFactorForInt_);
      if (readLines!=null){
        for (int i=0; i!=readLines.length;i++){
//        System.out.println(readLines[i]);
        //System.out.println("LineNumber: "+count+" Line: "+readLines[i]);
          float currentMz = 0;
          Float[] timeBorders = null;
          if (probe!=null){
            currentMz = probe.getEllipseMzPosition()-probe.getEllipseMzStretch()+i*mzAdduct;
            timeBorders = Calculator.calculateEllipseXBorderValues(currentMz, probe.getEllipseTimePosition(), probe.getEllipseMzPosition(), 
              probe.getEllipseTimeStretch(), probe.getEllipseMzStretch());
            if (probe.getLowerHardRtLimit()>=0 && timeBorders[0]<probe.getLowerHardRtLimit())
              timeBorders[0] = probe.getLowerHardRtLimit();
            if (probe.getUpperHardRtLimit()>=0 && timeBorders[1]>probe.getUpperHardRtLimit())
              timeBorders[1] = probe.getUpperHardRtLimit();
          }
          if (readLines[i]!=null&&readLines[i].length()>0){
            ByteBuffer buffer = ByteBuffer.wrap(Base64.decode(readLines[i]));
            while (buffer.hasRemaining()){
              int scanNumber = buffer.getInt();
              float intensity = buffer.getFloat();
              boolean intAdded = false;
              if (probe!=null){
                if (timeBorders[0]<=chrom.Value[scanNumber][0]&&chrom.Value[scanNumber][0]<=timeBorders[1]){
                  chrom.Value[scanNumber][1] += intensity;
                  intAdded = true;
                }  
              }else{
                chrom.Value[scanNumber][1] += intensity;
                intAdded = true;
              }
              if (intAdded && intensity>highestInt)
                // Juergen: I changed this from "highestInt = intensity"; I am not sure why here the individual
                // intensity was taken and not the summmed one?
                // If you change back - do the same for "precursorType
                highestInt = intensity;
//              float retentionTime = this.retentionTimes.get(new Integer(scanNumber));
//              System.out.println("Line: "+(count-this.lowestMz_)+" ScanNumber: "+scanNumber+"; Intensity: "+intensity);
            }
          }
        }
      }
    }
    chrom.setHighestIntensity(highestInt);
    if (msLevel==1 && sparseData_){
      chrom.doChromValueInterpolation(getRetentionTimes(1, false));
    }
    return chrom;
    
  }
  
  public CgChromatogram readJustIntensitiesOfInterest(Probe3D probe, int msLevel) throws CgException{
/*    CgChromatogram chrom = null;
    String[] readLines = this.getRawLines(probe.getEllipseMzPosition()-probe.getEllipseMzStretch(), probe.getEllipseMzPosition()+probe.getEllipseMzStretch());
    chrom = new CgChromatogram(this.numberOfScans);
    float mzAdduct = ((float)lowestResolution_/(float)multiplicationFactorForInt_);
    for (int i=0;i!=readLines.length;i++){
      float currentMz = probe.getEllipseMzPosition()-probe.getEllipseMzStretch()+i*mzAdduct;
      System.out.println(currentMz);
    }*/
    return this.readChromatogram(probe.getEllipseMzPosition()-probe.getEllipseMzStretch(), probe.getEllipseMzPosition()+probe.getEllipseMzStretch(),-1f,-1f,msLevel,probe);
  }
  
  /**
   * reads a chromatogram that contains only the contents of a certain m/z and time range - used for MS/MS peak quantitation
   * @param startMz startMz the start m/z value of the chromatogram
   * @param stopMz stopMz the stop m/z value of the chromatogram
   * @param startTime the start retention time of the chromatogram - use -1f for neglecting a start time
   * @param stopTime the stop retention time of the chromatogram - use -1f for neglecting a stop time
   * @param msLevel the MS-level of the crhomatogram
   * @return a chromatogram that contains only the contents of a certain m/z and time range - used for MS/MS peak quantitation
   * @throws CgException something wrong with the reading
   */
  public CgChromatogram readJustIntensitiesOfInterest(float startMz, float stopMz, float startTime, float stopTime, int msLevel) throws CgException{
    return this.readChromatogram(startMz, stopMz, startTime, stopTime, msLevel, null);
  }

  
  /** reads an m/z profile from the chrom file and smooths it*/
  protected Vector<CgChromatogram> readProfiles(Vector<CgProbe> probes, float mzTolerance, float timeTolerance,float maxTimeDeviation,
      float mzSmoothRange, int smoothRepeats, int msLevel) throws CgException{
    Hashtable<Integer,Float> startTime = new Hashtable<Integer,Float>();
    Hashtable<Integer,Float> stopTime = new Hashtable<Integer,Float>();
//    Hashtable<Integer,Float> startTimeBroad = new Hashtable<Integer,Float>();
//    Hashtable<Integer,Float> stopTimeBroad = new Hashtable<Integer,Float>();*/
    Hashtable<Integer,CgChromatogram> singleProfiles = new Hashtable<Integer,CgChromatogram>();
//    Hashtable<Integer,CgChromatogram> singleProfilesBroad = new Hashtable<Integer,CgChromatogram>();
//    Hashtable<Integer,Boolean> foundNotNull = new Hashtable<Integer,Boolean>();
    if (probes.size()>0){
      float mz = probes.get(0).Mz;
      // read lines of specific m/z range
      String[] readLines = this.getRawLines(mz-mzTolerance, mz+mzTolerance, msLevel);
      if (readLines!=null){      
        for (int i=0;i!=probes.size();i++){
          CgProbe probe = probes.get(i);
          startTime.put(i, probe.Peak-timeTolerance);
          stopTime.put(i, probe.Peak+timeTolerance);
//          startTimeBroad.put(i, probe.Peak-maxTimeDeviation);
//          stopTimeBroad.put(i, probe.Peak+maxTimeDeviation);

          CgChromatogram profile = new CgChromatogram(readLines.length);
          CgChromatogram profileBroad = new CgChromatogram(readLines.length);
          for (int j=0;j!=readLines.length;j++){
            profile.Value[j][1] = 0;
            profileBroad.Value[j][1] = 0;
          }
          singleProfiles.put(i, profile);
//          singleProfilesBroad.put(i, profileBroad);
//          foundNotNull.put(i, false);
        }
        // decodes the lines and writes it into a profile (similar to chromatogram but m/z direction)
        for (int i=0;i!=readLines.length;i++){
          for (int j=0;j!=singleProfiles.size();j++){
            float value = mz-mzTolerance+(i*(((float)this.getLowestResolution_())/this.getMultiplicationFactorForInt_()));
            singleProfiles.get(j).Value[i][0]=value;
//            singleProfilesBroad.get(j).Value[i][0]=value;
          }
          if (readLines[i]!=null&&readLines[i].length()>0){
            ByteBuffer buffer = ByteBuffer.wrap(Base64.decode(readLines[i]));
            while (buffer.hasRemaining()){
              int scanNumber = buffer.getInt();
              float retentionTime = this.retentionTimesOriginal_.get(new Integer(scanNumber));
              float intensity = buffer.getFloat();
              for (int j=0;j!=singleProfiles.size();j++){
                if (startTime.get(j)<retentionTime&&retentionTime<stopTime.get(j)){
                  singleProfiles.get(j).Value[i][1]+=intensity;
//                  if (intensity>0)
//                    foundNotNull.put(j, true);
                }
//                if (!foundNotNull.get(j)&&startTimeBroad.get(j)<retentionTime&&stopTime.get(j)<stopTimeBroad.get(j))
//                  singleProfilesBroad.get(j).Value[i][1]+=intensity;
              }
            }
          }
//          for (int j=0;j!=singleProfiles.size();j++){
//            if (!foundNotNull.get(j))
//              singleProfiles.put(j, singleProfilesBroad.get(j));
//          }
        }

        
        
//        int deviationCount = 0;
//        boolean justZeroValues = true;
////        this.extractIntensityValues(readLines,singleProfiles,mz, mzTolerance, startTime, stopTime, timeTolerance,0);
//        while (justZeroValues&&deviationCount*timeTolerance<maxTimeDeviation){
//          if (this.extractIntensityValues(readLines,singleProfiles,mz, mzTolerance, startTime, stopTime, timeTolerance,deviationCount*(-1)))
//              justZeroValues = false;
//          if (justZeroValues&&deviationCount != 0){
//            if (this.extractIntensityValues(readLines,singleProfiles,mz, mzTolerance, startTime, stopTime, timeTolerance,deviationCount))
//              justZeroValues = false;            
//          }
//          deviationCount++;
//        }
      }
    }
    // smooths the profiles
    Vector<CgChromatogram> profiles = smoothSingleProfiles(probes,singleProfiles,mzSmoothRange,smoothRepeats,startTime,stopTime);    
    return profiles;
  }
  
  protected Vector<CgChromatogram> smoothSingleProfiles(Vector<CgProbe> probes, Hashtable<Integer,CgChromatogram> singleProfiles, float mzSmoothRange,
      int smoothRepeats, Hashtable<Integer,Float> startTime, Hashtable<Integer,Float> stopTime){
    Vector<CgChromatogram> profiles = new Vector<CgChromatogram>(); 
    for (int i=0;i!=probes.size();i++){
      CgChromatogram cx = singleProfiles.get(i);
      cx.isProfile_ = true;
      cx.Mz = probes.get(i).Peak;
      cx.LowerMzBand = startTime.get(i);
      cx.UpperMzBand = stopTime.get(i);
//      cx.preSmooth();
      cx.Smooth(mzSmoothRange, smoothRepeats);
//      cx.Smooth(0.005f, 2,false);
//      cx.Smooth(0.005f, 3);
      profiles.add(cx);
    }    
    return profiles;
  }
  

//  private boolean extractIntensityValues(String[] readLines,Hashtable<Integer,CgChromatogram> singleProfiles,float mz, float mzTolerance,
//      Hashtable<Integer,Float> startTime,Hashtable<Integer,Float> stopTime, float timeTolerance,int deviationCount){
//    boolean foundNonZeroValue = false;
//    for (int i=0;i!=readLines.length;i++){
//      for (int j=0;j!=singleProfiles.size();j++){
//        singleProfiles.get(j).Value[i][0]=mz-mzTolerance+(i*(((float)this.getLowestResolution_())/this.getMultiplicationFactorForInt_()));
//      }
//      if (readLines[i]!=null&&readLines[i].length()>0){
//        ByteBuffer buffer = ByteBuffer.wrap(Base64.decode(readLines[i]));
//        while (buffer.hasRemaining()){
//          int scanNumber = buffer.getInt();
//          float retentionTime = this.retentionTimes.get(new Integer(scanNumber));
//          float intensity = buffer.getFloat();
//          for (int j=0;j!=singleProfiles.size();j++){
//            float start = startTime.get(j)+deviationCount*timeTolerance;
//            float stop = stopTime.get(j)+deviationCount*timeTolerance;
//            if (start<retentionTime&&retentionTime<stop){
//              singleProfiles.get(j).Value[i][1]+=intensity;
//              if (intensity>0)
//                foundNonZeroValue = true;
//            }
//          }
//        }
//      }
//    }
//    return foundNonZeroValue;
//  }  
  
  
  
  public String[] getRawLines(float from, float to) throws CgException{
    int lineInFile = this.getLineNumber(from); 
    if (lineInFile<0) lineInFile = 0;
    int stopLine = this.getLineNumber(to);
    String[] readLines;
    if (this.chromCache_){
      readLines = this.readCachedLines(lineInFile, stopLine);
    }else{
      try{
        readLines = this.readLines(lineInFile, stopLine);
      }catch(IndexFileException ifx){
        throw new CgException (ifx.getMessage());
      }
    }
    return readLines;
  }
  
  private int getLineNumber(float mzValue){
    return Math.round(Calculator.roundFloat(mzValue*(multiplicationFactorForInt_/lowestResolution_), 0,BigDecimal.ROUND_DOWN))-this.lowestMz_/this.lowestResolution_;    
  }
  
  public String[] getRawLines(float from, float to, int msLevel) throws CgException{
    return getRawLines(from,to,msLevel,10);
  }
  
  public String[] getRawLines(float from, float to, int msLevel, int lowestResolutionMultiplicationFactor) throws CgException{
    if (msLevel==1){
      if (msmsType_!=null && msmsType_.equalsIgnoreCase(CHROMATOGRAM_HEADER_FILE_MSMS_TYPE_FULL)&&this.lastReadMsLevel_!=1)
        try {
          setGlobalVariblesForSuperClass(1);
        }
        catch (IndexFileException e) {
          throw new CgException(e.getMessage());
        }
      return getRawLines(from, to);
    }
    else if (msLevel>1 && msLevel<=this.highestMsLevel_){
      if (msmsType_!=null && msmsType_.equalsIgnoreCase(CHROMATOGRAM_HEADER_FILE_MSMS_TYPE_PRECURSOR)){
//        Vector<String> msmsLines = getMsMsSpectra(from, to, msLevel);
        if (!msnSpectraCache_.containsKey(msLevel)) throw new CgException("The MS-level of \""+msLevel+"\" is read in the cache! For "+CHROMATOGRAM_HEADER_FILE_MSMS_TYPE_PRECURSOR+" files, the method prepareMSnSpectraCache has to be called before");
        Hashtable<Integer,String> msmsLines = msnSpectraCache_.get(msLevel);
        List<Integer> consScanNrs = new ArrayList<Integer>(msmsLines.keySet());
        Collections.sort(consScanNrs);
        Vector<String> spectra = new Vector<String>();
        for (Integer conScanNr : consScanNrs) spectra.add(msmsLines.get(conScanNr));
        return translateSpectraToChroms(spectra,msLevel,lowestResolutionMultiplicationFactor);
        
      }else if (msmsType_!=null && msmsType_.equalsIgnoreCase(CHROMATOGRAM_HEADER_FILE_MSMS_TYPE_FULL)){
        try{
          setGlobalVariblesForSuperClass(msLevel);

          return getRawLines(from, to);
        } catch (IndexFileException ifx){
          throw new CgException(ifx.getMessage());
        }
      }
      throw new CgException("The type of your MS/MS Spectra is not supported");
    }else if (msLevel>this.highestMsLevel_)
      throw new CgException("The MS-level of \""+msLevel+"\" is not supported by your file! The highest MS Level available is: "+highestMsLevel_);
    else
      throw new CgException("Illegal value for MS-Level: "+msLevel); 
  }

  /**
   * returns the MS/MS spectra of a certain time m/z- and/or time-range 
   * @param from the start precursor m/z
   * @param to the stop precursor m/z
   * @param fromTime the start retention time - use -1f for neglecting a start time
   * @param toTime the stop retention time - use -1f for neglecting a start time
   * @param msLevel the MS-level
   * @return MS/MS spectra of a certain time m/z- and/or time-range; first key: the MS-level
   * @throws CgException something wrong with the reading
   */
  public Hashtable<Integer,Vector<String>> getMsMsSpectra(float from, float to, float fromTime, float toTime) throws CgException{
    Hashtable<Integer,Vector<String>> spectralLines = new Hashtable<Integer,Vector<String>>();
    if (msmsType_ != null){
      for (int i=2; i<= this.highestMsLevel_; i++){
        try{
          Vector<String> msmsLines = getMsMsSpectra(from, to, fromTime, toTime, i);
          if (msmsLines.size()>0)
            spectralLines.put(i, msmsLines);
        } catch (CgException cex){
          if (!cex.getMessage().equalsIgnoreCase(NO_MSMS_SPECTRA))
            throw cex;
        }
      }
    }
    if (spectralLines.size()<1)
      throw new CgException(NO_MSMS_SPECTRA);
    return spectralLines;
  }
  
  /**
   * returns the MS/MS spectra of a certain time m/z- and/or time-range 
   * @param from the start precursor m/z
   * @param to the stop precursor m/z
   * @param fromTime the start retention time - use -1f for neglecting a start time
   * @param toTime the stop retention time - use -1f for neglecting a start time
   * @param msLevel the MS-level
   * @return MS/MS spectra of a certain time m/z- and/or time-range
   * @throws CgException something wrong with the reading
   */
  private Vector<String> getMsMsSpectra(float from, float to, float fromTime, float toTime, int msLevel) throws CgException{
    Vector<String> msmsLines = new Vector<String>();
    try{
      if (!cachedMsMsIndices_.containsKey(msLevel)) readMsMsIndexFile(msLevel);
      int fromLine = this.getLineNumber(from);
      fromLine = fromLine-(fromLine%multiplicationFactorForInt_/lowestResolution_);
      boolean finishedReading = false;
      boolean foundStartLine = false;
      boolean foundEndLine = false;
      int indexNumber = getIndexFileNumber(fromLine,numberOfLinesForIndex_);
//      int currentLine = 0;
      long bytesToSkip = 0;
      if (cachedMsMsLines_.get(msLevel).size()>indexNumber){
//        currentLine = cachedMsMsLines_.get(msLevel).get(indexNumber);
        bytesToSkip = cachedMsMsIndices_.get(msLevel).get(indexNumber);
        RandomAccessFile reader = null; 
        BufferedReader lreader = null;
        String line = "";
        String mzString = "";
        try {
          reader = new RandomAccessFile(msmsChromFiles_.get(msLevel-1),"r");
          reader.seek(bytesToSkip);
          lreader = new BufferedReader(new FileReader(reader.getFD()));
          while (!finishedReading){
            line = lreader.readLine();
            if (line!=null && line.startsWith(">")){
              mzString = line.substring(1);
              if (mzString.indexOf(" ")!=-1) mzString = mzString.substring(0,mzString.indexOf(" "));
              float mzValue = Float.parseFloat(mzString);
              if (mzValue>=from) foundStartLine = true;
              if (mzValue>to) foundEndLine = true;
              //if there are time constraints, it might be that one m/z value does not contain any spectra
              if ((fromTime>0 || toTime>0) && msmsLines.size()>0 && msmsLines.get(msmsLines.size()-1).startsWith(">")){
                msmsLines.remove(msmsLines.size()-1);
              }
            }
            if (foundEndLine || line==null) finishedReading = true;
            if (line!=null && foundStartLine&&!foundEndLine){
              if ((fromTime>0 || toTime>0)){
                if (line.startsWith(">"))
                  msmsLines.add(line);
                else {
                  int spaceIndex = line.indexOf(" ");
                  int scanNumber = Integer.parseInt(line.substring(0,spaceIndex));
                  int consecutiveNumber = fromScanToConsecutiveNumber_.get(msLevel).get(scanNumber);
                  float retTime = msmsRetentionTimes_.get(msLevel).get(consecutiveNumber);
                  boolean addLine = true;
                  if (fromTime>0 && retTime<fromTime) addLine = false;
                  if (toTime>0 && retTime>toTime) addLine = false;
                  if (addLine)
                    msmsLines.add(line);
                }
              }else
                msmsLines.add(line);
            }
//            currentLine++;
          }
        }
        catch (IOException e) {
          throw new IndexFileException(e.getMessage());
        }finally{
          try{
            if (lreader!=null){
              lreader.close();
              if (reader!=null)
                reader.close();
            }
          }catch(IOException e){};
        }
      }
    } catch (IndexFileException ifx){
      throw new CgException(ifx.getMessage());
    }
    if (msmsLines==null || msmsLines.size()<1) throw new CgException(NO_MSMS_SPECTRA);
    return msmsLines;
  }
  
  public String[] translateSpectraToChroms(Vector<String> spectra, int msLevel, int lowestResolutionMultiplicationFactor)throws CgException{
    if (spectra==null || spectra.size()<1) throw new CgException("There are no MS/MS spectra for this precursor!");
    float lowestMzValue = Float.MAX_VALUE;
    float highestMzValue = 0;
    int lowestScanNumber = Integer.MAX_VALUE;
    int highestScanNumber = 0;
    Hashtable<Integer,CgScan> scans = new Hashtable<Integer,CgScan>();
    Hashtable<Integer,Float> retTimes = msmsRetentionTimes_.get(msLevel);
    for (int i=0;i!=spectra.size();i++){
      String line = spectra.get(i);
      if (!line.startsWith(">")){
        int spaceIndex = line.indexOf(" ");
        int scanNumber = Integer.parseInt(line.substring(0,spaceIndex));
        int consecutiveNumber = fromScanToConsecutiveNumber_.get(msLevel).get(scanNumber);
        if (consecutiveNumber<lowestScanNumber) lowestScanNumber = consecutiveNumber;
        if (consecutiveNumber>highestScanNumber) highestScanNumber = consecutiveNumber;
        String spectrum = line.substring(spaceIndex+1);
        FloatBuffer buffer = ByteBuffer.wrap(Base64.decode(spectrum)).asFloatBuffer();
        int iItems = buffer.limit();
        Vector<Float> mzValues = new Vector<Float>(); 
        Vector<Float> intensities = new Vector<Float>(); 
        for(int iItem = 0; iItem < iItems; iItem++){
          if (iItem%2==0) mzValues.add(buffer.get());
          else intensities.add(buffer.get());
        }
        CgScan scan = new CgScan(mzValues.size(), scanNumber, 2, retTimes.get(consecutiveNumber), -1f, -1f, -1f, -1f,-1f,CgDefines.POLARITY_NO);
        for (int j=0;j!=mzValues.size();j++){
          scan.Scan[j][0] = mzValues.get(j);
          scan.Scan[j][1] = intensities.get(j);
          if (scan.Scan[j][0]<lowestMzValue) lowestMzValue = scan.Scan[j][0];
          if (scan.Scan[j][0]>highestMzValue) highestMzValue = scan.Scan[j][0];
        }
        scans.put(consecutiveNumber, scan);
      }
    }
    int lowMzValue = (int)Calculator.roundFloat(lowestMzValue, 0,BigDecimal.ROUND_DOWN);
    int highMzValue =  (int)Calculator.roundFloat(highestMzValue, 0,BigDecimal.ROUND_UP);
    int nrOfLines = (highMzValue-lowMzValue)*lowestResolutionMultiplicationFactor;
    String[] chroms = new String[nrOfLines];
    // this could made more effective with GetIntensitiyArray, but the MS2 spectra are normally not so many
    for (int i=0;i!=nrOfLines;i++){
      float fromMz = (float)lowMzValue+((float)i)/((float)lowestResolutionMultiplicationFactor);
      float toMz = ((float)lowMzValue+(float)(i+1)/((float)lowestResolutionMultiplicationFactor));
      Vector<Integer> scanNumbers = new Vector<Integer>();
      Vector<Float> intensities = new Vector<Float>();
      for (int j=0; j<retTimes.size(); j++){
        if (scans.containsKey(j)){
          CgScan scan = scans.get(j);
          float intensity = scan.GetIntensity(fromMz, toMz);
          if (intensity>0){
            scanNumbers.add(j);
            intensities.add(intensity);
          }
        }
      }
      String chromString = "";
      if (scanNumbers.size()>0){
        ByteBuffer buffer = ByteBuffer.allocate(scanNumbers.size() * 2 * 4);
        for (int j=0;j!=scanNumbers.size();j++){
          buffer.putInt(scanNumbers.get(j).intValue());
          buffer.putFloat(intensities.get(j).floatValue());
        }
        chromString = String.valueOf(Base64.encode(buffer.array()));
      }
      chroms[i] = chromString;
    }
    ms2ExtremeValues_[0] = lowMzValue;
    ms2ExtremeValues_[1] = highMzValue;
    ms2ExtremeValues_[2] = lowestScanNumber-5;
    if (ms2ExtremeValues_[2]<0)ms2ExtremeValues_[2]=0; 
    ms2ExtremeValues_[3] = highestScanNumber+5;
    if (ms2ExtremeValues_[3]>(retTimes.size()-1)) ms2ExtremeValues_[3] = retTimes.size()-1;
    return chroms;
  }
  
  private void  setGlobalVariblesForSuperClass(int msLevel) throws IndexFileException{
    if (!cachedMsMsIndices_.containsKey(msLevel)) readMsMsIndexFile(msLevel);
    if (lastReadMsLevel_!=msLevel){
      indexFilePath_ = msmsIndexFiles_.get(msLevel-1);
      dataFilePath_ = msmsChromFiles_.get(msLevel-1);
      numberOfLines_ = msmsNrOfLines_.get(msLevel);
      cachedLine_ = cachedMsMsLines_.get(msLevel);
      cachedIndex_ = cachedMsMsIndices_.get(msLevel);            
    }
    lastReadMsLevel_ = msLevel;
  } 
  /**
   * reads the lines for the chromatogram from the cache input file
   * @param start start line to read
   * @param stop stop line to read
   * @return the necessary lines as string array
   */
  private String[] readCachedLines(int start, int stop){
    if (start<0) start = 0;
    if (stop>this.cachedChromatogramFile.length)stop = this.cachedChromatogramFile.length;
    String[] lines = new String[0];
    if (stop>start){
      lines = new String[stop-start];
      for (int i=0; i!=(stop-start);i++){
        lines[i] = this.cachedChromatogramFile[i+start]; 
      }
    }
    return lines;
  }
  /**
   * reads the whole chromatogram file into a String array
   * @throws IndexFileException
   */
  private void readWholeChromatogramIntoFile() throws IndexFileException{
    this.cachedChromatogramFile = this.readLines(0, (this.highestMz_-this.lowestMz_));
  }

  /**
   * 
   * @return the original scan numbers
   */
  public Vector<Integer> getOriginalScanNumbers()
  {
    return originalScanNumbers;
  }

  public Hashtable<Integer,Float> getRetentionTimes()
  {
    return retentionTimes_;
  }
  
  /**
   * @return returns the original retention time values as obtained from the chrom file
   */
  public Hashtable<Integer,Float> getRetentionTimesOriginal()
  {
    return retentionTimesOriginal_;
  }
  
  /**
   *  @return returns the retention times for the chrom values
   */
  public Hashtable<Integer,Float> getRetentionTimes(int msLevel)
  {
    return getRetentionTimes(msLevel,false);
  }
  
  /**
   * 
   * @param msLevel the MS level for the retention times
   * @param useOriginal use the original RT values (without interpolation) - only applicable for MS1 mode
   * @return the retention times for the chrom values
   */
  public Hashtable<Integer,Float> getRetentionTimes(int msLevel, boolean useOriginal)
  {
    if (msLevel==1){
      if (useOriginal) return this.retentionTimesOriginal_;
      else return getRetentionTimes();
    }
    return msmsRetentionTimes_.get(msLevel);
  }
  
  /**
   * 
   * @return the hash of the retention times of the chrom values
   */
  public Hashtable<Integer,Float> getMsmsRetentionTimes() {
    return msmsScanNrToRetentionTime_;
  }
  
  public int getNumberOfScans(){
    return this.numberOfScans;
  }
  
  
  public int findBaseScanIndex(int scanNumber)
  {
    int baseScanIndex = 0;
    for (int i = 0; i < originalScanNumbers.size(); i++) {
      if (scanNumber>=(originalScanNumbers.get(i)).intValue()) baseScanIndex++;
      else{
        if (baseScanIndex!=0){
          baseScanIndex--;
          return (baseScanIndex);
        }
      }
    }
    if (baseScanIndex==originalScanNumbers.size()){
      baseScanIndex--;
      return (baseScanIndex);
    }
    return baseScanIndex;
  }

  public int getHighestMz_()
  {
    return highestMz_;
  }

  public int getLowestMz_()
  {
    return lowestMz_;
  }

  public int getLowestResolution_()
  {
    return lowestResolution_;
  }

  public int getMultiplicationFactorForInt_()
  {
    return multiplicationFactorForInt_;
  }

  @SuppressWarnings({ "unchecked", "rawtypes" })
  public static Vector translateBase64ToValues(String encodedLine){
    Vector values = new Vector();
    if (encodedLine!=null&&encodedLine.length()>0){
      ByteBuffer buffer = ByteBuffer.wrap(Base64.decode(encodedLine));
      while (buffer.hasRemaining()){
        int scanNumber = buffer.getInt();
        float intensity = buffer.getFloat();
        values.add(new Integer(scanNumber));
        values.add(new Float(intensity));
      }
    }
    return values;
  }
  
  private void readMsMsIndexFile(int msLevel) throws IndexFileException {
    Hashtable<Integer,Integer> cachedLine = new Hashtable<Integer,Integer>();
    Hashtable<Integer,Long> cachedIndex = new Hashtable<Integer,Long>();
    super.readIndexFileWhole(msmsIndexFiles_.get(msLevel-1), cachedLine, cachedIndex);
    cachedMsMsLines_.put(msLevel, cachedLine);
    cachedMsMsIndices_.put(msLevel, cachedIndex);
  }
  
  public Hashtable<Integer,Integer> getFromScanToConsecutive(int msLevel){
    return fromScanToConsecutiveNumber_.get(msLevel);
  }
  
  public int getHighestMsLevel(){
    return highestMsLevel_;
  }
  
  public String getMsmsType(){
    return this.msmsType_;
  }
  
  /**
   * This method returns the m/z and scanNumber borders of the
   * 
   * @return
   */
  public int[] getBordersOfLastMs2Extraction(){
    return this.ms2ExtremeValues_;
  }
  
  /**
   * this method generates three hash tables and returns them; the key in all of them is the scan number
   * the return vector at position one contains the spectra;
   * the return vector at position two contains a Vector of type Double, containing the precursor masses
   * the return vector at position three is a lookup showing from which MS-level the scan originates 
   * @param spectra 
   * @return
   */
  @SuppressWarnings("rawtypes")
  public Vector<Hashtable> getRtNrSpectrumHash(Hashtable<Integer,Vector<String>> spectra){
    Hashtable<Integer,String> spectraHash = new Hashtable<Integer,String>();
    Hashtable<Integer,Vector<Double>> precursorHash = new Hashtable<Integer,Vector<Double>>();
    Hashtable<Integer,Integer> levelHash = new Hashtable<Integer,Integer>();
    Vector<Double> precursorMasses = null;
    String precursorMassString = null;
    for (Integer msLevel : spectra.keySet()){
      Vector<String> spectraOneLevel = spectra.get(msLevel);
      for (int i=0;i!=spectraOneLevel.size();i++){
        String line = spectraOneLevel.get(i);
        if (line.startsWith(">")){
          precursorMassString = line.substring(1);
          precursorMasses = new Vector<Double>();
          for (String mass : precursorMassString.split(" ")){
            precursorMasses.add(new Double(mass));
          }
        } else {
          int spaceIndex = line.indexOf(" ");
          int scanNumber = Integer.parseInt(line.substring(0,spaceIndex));
          String spectrum = line.substring(spaceIndex+1);
          spectraHash.put(scanNumber, spectrum);
          precursorHash.put(scanNumber, precursorMasses);
          levelHash.put(scanNumber, msLevel);
        }
      }
    }
    Vector<Hashtable> results = new Vector<Hashtable>();
    results.add(spectraHash);
    results.add(precursorHash);
    results.add(levelHash);
    return results;
  }
  
  /**
   * removes the spectra stored in the cache - reason: if new spectra are extracted,
   * the old ones have to be removed
   */
  private void cleanMsLevelCache(){
    msnSpectraCache_ = new Hashtable<Integer,Hashtable<Integer,String>>();
  }
  
  /**
   * caches the spectra of a certain precursor mass
   * @param fromMz lower m/z threshold for precursor m/z
   * @param toMz upper m/z threshold for precursor m/z
   * @param fromTime the start retention time for spectra caching - use -1f for neglecting a start time 
   * @param toTime the stop retention time for spectra caching - use -1f for neglecting a start time 
   * @param msLevel MSn level on which the extraction shall be performed
   * @param minIntsForNoiseRemoval minimum amount of detected signals to start a noise removal
   * @return true if there are spectra available for this MS-level
   * @throws CgException
   */
  private boolean cacheSpectraOfMsLevel(float fromMz, float toMz, float fromTime, float toTime, int msLevel,
      int minIntsForNoiseRemoval) throws CgException {
    if (!checkIfMSLevelIsThere(msLevel)) return false;
    if (msmsType_.equalsIgnoreCase(CHROMATOGRAM_HEADER_FILE_MSMS_TYPE_FULL)) return true;
    if (msmsType_.equalsIgnoreCase(CHROMATOGRAM_HEADER_FILE_MSMS_TYPE_PRECURSOR)){
      try{
        Vector<String> msmsSpectra = getMsMsSpectra(fromMz, toMz, fromTime, toTime, msLevel);
        if (msmsSpectra.size()<1) return false;
        Hashtable<Integer,String> foundConsecutiveScans = new Hashtable<Integer,String>();
        Hashtable<Integer,Float> noiseLevels = new Hashtable<Integer,Float>();
        for (int i=0;i!=msmsSpectra.size();i++){
          String line = msmsSpectra.get(i);
          if (!line.startsWith(">")){
            int spaceIndex = line.indexOf(" ");
            int scanNumber = Integer.parseInt(line.substring(0,spaceIndex));
            int consecutiveNumber = fromScanToConsecutiveNumber_.get(msLevel).get(scanNumber);
            String spectrum = line.substring(spaceIndex+1);
            float noise = 0f;
            if (minIntsForNoiseRemoval>0)
              noise = estimateSpectrumNoise(spectrum, minIntsForNoiseRemoval);
            foundConsecutiveScans.put(consecutiveNumber, line);
            noiseLevels.put(consecutiveNumber, noise);
          }
        }
        msnSpectraCache_.put(msLevel,foundConsecutiveScans);
        msnSpectraNoise_.put(msLevel, noiseLevels);
        return true;
      } catch (CgException cgx){
        return false;
      }
    }
    return false;
  }
  
  /**
   * @param spectrum Base64 encoded spectrum
   * @param minIntsForNoiseRemoval minimum amount of detected signals to start a noise removal
   * @return the intensity of the proposed noise level
   */
  private float estimateSpectrumNoise(String spectrum, int minIntsForNoiseRemoval){
    float noise = 0f;
    FloatBuffer buffer = ByteBuffer.wrap(Base64.decode(spectrum)).asFloatBuffer();
    int iItems = buffer.limit();
    float intensity = 0;
    Vector<Float> intensities = new Vector<Float>();
    for(int iItem = 0; iItem < iItems; iItem++){
      if (iItem%2==0) buffer.get();
      else{
        intensity = buffer.get();
        if (intensity>0) intensities.add(intensity);
      }
    }
    if (intensities.size()>minIntsForNoiseRemoval){
      Collections.sort(intensities);      
//      int position = (intensities.size()*7)/10;
//      noise = intensities.get(position);
      float delta = 0f;
      float noiseImprovement = 1.7f;
      for (int i=1;i!=(intensities.size()/2);i++){
        delta += intensities.get(i)-intensities.get(i-1);
      }
      delta = delta/(float)(intensities.size()/2);
      float[][] matrixValues = new float[intensities.size()-1][2];
      float[][] intensityValues = new float[intensities.size()-1][1];
      for (int j=0;j!=intensities.size()-1;j++){
        matrixValues[j][0] = (j+1);
        matrixValues[j][1] = 1f;
        intensityValues[j][0] = intensities.get(j);
      }
      
      
      for (int i=(intensities.size()/2);i!=intensities.size();i++){
        FloatMatrix matrix = new FloatMatrix(matrixValues);
        matrix.m = i-1;
        FloatMatrix transposed = matrix.transpose();
        FloatMatrix product = transposed.times(matrix);
        FloatMatrix intensityMatrix = new FloatMatrix(intensityValues);
        intensityMatrix.m = i-1;
        FloatMatrix result = product.inverse().times(transposed).times(intensityMatrix);
        noise = (float)i*result.A[0][0]+result.A[1][0];
        if (intensities.get(i)>(noise*noiseImprovement)){
          break;
        }
      }
    }
    return noise;
  }

  /**
   * caches the spectra of a certain precursor mass
   * @param from lower m/z threshold for precursor m/z
   * @param to upper m/z threshold for precursor m/z
   * @param minIntsForNoiseRemoval minimum number of detected signals to start a noise removal
   * @return
   * @throws CgException
   */
  public Hashtable<Integer,Boolean> prepareMSnSpectraCache(float from, float to, int minIntsForNoiseRemoval) throws CgException{
    return prepareMSnSpectraCache(from, to, -1f, -1f, minIntsForNoiseRemoval);
  }
  
  /**
   * caches the spectra of a certain precursor mass and/or retention time range
   * @param fromMz lower m/z threshold for precursor m/z
   * @param toMz upper m/z threshold for precursor m/z
   * @param fromTime the start retention time of the chromatogram - use -1f for neglecting a start time
   * @param toTime the stop retention time of the chromatogram - use -1f for neglecting a start time
   * @param minIntsForNoiseRemoval minimum amount of detected signals to start a noise removal
   * @return spectra of a certain precursor mass and/or retention time range
   * @throws CgException something wrong with the reading
   */
  public Hashtable<Integer,Boolean> prepareMSnSpectraCache(float fromMz, float toMz, float fromTime, float toTime,
      int minIntsForNoiseRemoval) throws CgException{
    Hashtable<Integer,Boolean> availableLevels = new Hashtable<Integer,Boolean>();
    cleanMsLevelCache();
    if (msmsType_ != null){
      for (int i=2; i<= this.highestMsLevel_; i++){
        boolean isThere = cacheSpectraOfMsLevel(fromMz, toMz, fromTime, toTime, i, minIntsForNoiseRemoval);
        if (isThere) availableLevels.put(i, true);
      }
    }
    return availableLevels;
  }

  /**
   * ATTENTION: prepareMSnSpectraCache has to be called before
   * queries the data if there are spectra in a certain time range
   * @param startTime lower time threshold in seconds
   * @param stopTime upper time threshold in seconds
   * @param msLevel MSn level on which the check shall be performed
   * @return true if there is any data at this time range
   * @throws CgException
   */
  public boolean areMSnSpectraInThisRegion(float startTime, float stopTime, int msLevel) throws CgException{
    if (!checkIfMSLevelIsThere(msLevel)) return false;
    if (msmsType_.equalsIgnoreCase(CHROMATOGRAM_HEADER_FILE_MSMS_TYPE_FULL)) return true;
    if (msmsType_.equalsIgnoreCase(CHROMATOGRAM_HEADER_FILE_MSMS_TYPE_PRECURSOR)){
      if (!msnSpectraCache_.containsKey(msLevel)) return false;
      Hashtable<Integer,Float> retTimes = msmsRetentionTimes_.get(msLevel);
      for (int consScanNumber : this.msnSpectraCache_.get(msLevel).keySet()){
        double retTime = retTimes.get(consScanNumber);
        if (startTime<=retTime && retTime <= stopTime) return true;
      }
    }
    return false;
  }
  
  /**
   * ATTENTION: prepareMSnSpectraCache has to be called before
   * queries the data if there are spectra in a certain time range
   * @param startTime lower time threshold in seconds
   * @param stopTime upper time threshold in seconds
   * @param msLevel MSn level on which the check shall be performed
   * @return the amount of MSn spectra in a certain time range
   * @throws CgException
   */
  public int countMSnSpectraOfRegion(float startTime, float stopTime, int msLevel) throws CgException{
    int count = 0;
    if (!checkIfMSLevelIsThere(msLevel)) return count;
    if (msmsType_.equalsIgnoreCase(CHROMATOGRAM_HEADER_FILE_MSMS_TYPE_FULL)) return count;
    if (msmsType_.equalsIgnoreCase(CHROMATOGRAM_HEADER_FILE_MSMS_TYPE_PRECURSOR)){
      if (!msnSpectraCache_.containsKey(msLevel)) return count;
      Hashtable<Integer,Float> retTimes = msmsRetentionTimes_.get(msLevel);
      for (int consScanNumber : this.msnSpectraCache_.get(msLevel).keySet()){
        double retTime = retTimes.get(consScanNumber);
        if (startTime<=retTime && retTime <= stopTime) count++;
      }
    }
    return count;
  }
  
  /**
   * checks if the acquired MSn level exists in the data
   * @param msLevel the MSn level to be checked
   * @return true if the MSn level is available
   * @throws CgException
   */
  private boolean checkIfMSLevelIsThere(int msLevel) throws CgException {
    if (msLevel<1) throw new CgException("The acquired MS level must be greater or equal 1 for MS level caching");
    if (msmsType_ == null) return false;
    if (msLevel>this.highestMsLevel_) return false;
    return true;
  }
  
  
  /**
   * ATTENTION: prepareMSnSpectraCache has to be called before
   * extracts intensity information
   * @param levels for which MSn levels should the base peak values be extracted
   * @param startTime lower time threshold in seconds
   * @param stopTime upper time threshold in seconds
   * @param mzTolerance +/- MSn range for the base peak
   * @return hash table containing as key the MSn level and as value the base peak intensity
   * @throws CgException
   */
  public Hashtable<Integer,Float> extractBasePeakValues(Vector<Integer> levels, float startTime, float stopTime, float mzTolerance) throws CgException {
    Hashtable<Integer,Float> basePeaks = new Hashtable<Integer,Float>();
    for (Integer msLevel : levels){
      float basePeakArea = 0f;
      if (msmsType_!=null && msmsType_.equalsIgnoreCase(CHROMATOGRAM_HEADER_FILE_MSMS_TYPE_PRECURSOR)){
        Hashtable<Integer,String> allSpectra = msnSpectraCache_.get(msLevel);
        Hashtable<Integer,Float> retTimes = msmsRetentionTimes_.get(msLevel);
        float lowestMz = Float.MAX_VALUE;
        float highestIntensity = 0f;
        Hashtable<Integer,Vector<Float>> mzs = new Hashtable<Integer,Vector<Float>>();
        Hashtable<Integer,Vector<Float>> intensities = new Hashtable<Integer,Vector<Float>>();
        Hashtable<Integer,Integer> currentItems = new Hashtable<Integer,Integer>();
        Hashtable<Integer,Float> maxMZs = new Hashtable<Integer,Float>();
        int count = 0;
        List<Float> mzsSorted = new ArrayList<Float>();
        for (int consScanNumber : allSpectra.keySet()){
          double retTime = retTimes.get(consScanNumber);
          if (startTime<=retTime && retTime <= stopTime){
            String spectrum = allSpectra.get(consScanNumber);
            spectrum = spectrum.substring(spectrum.indexOf(" ")+1);
            FloatBuffer buffer = ByteBuffer.wrap(Base64.decode(spectrum)).asFloatBuffer();
            int limit = buffer.limit();
            float mz = 0f;
            float intensity = 0f;
            Vector<Float> mzValues = new Vector<Float>();
            Vector<Float> intensityValues = new Vector<Float>();
            for(int iItem = 0; iItem < limit; iItem++){
              if (iItem%2==0) mz = buffer.get();
              else{
                intensity = buffer.get();
                if (intensity>(highestIntensity/10000f)){
                  mzValues.add(mz);
                  intensityValues.add(intensity);
                  if (mz<lowestMz) lowestMz = mz;
                  if (intensity>highestIntensity) highestIntensity = intensity;
                  mzsSorted.add(mz);
                }
              }
            }
            mzs.put(count, mzValues);
            intensities.put(count, intensityValues);
            currentItems.put(count, 0);
            maxMZs.put(count, Float.MAX_VALUE);
            count++;
          }
        }
        lowestMz = Calculator.roundFloat(lowestMz, 0, BigDecimal.ROUND_DOWN);
        Collections.sort(mzsSorted);
        for (float currentMz:mzsSorted){
          float sumArea = 0;
          for (int i=0;i!=mzs.size();i++){
            Vector<Float> mzValues = mzs.get(i);
            Vector<Float> intensityValues = intensities.get(i);
            int currentItem = currentItems.get(i);
            int selectItem = currentItem-1;
            while (selectItem>-1 && (mzValues.get(selectItem)>=(currentMz-mzTolerance))){
              if (mzValues.get(selectItem)<(currentMz+mzTolerance)) sumArea += intensityValues.get(selectItem);
              selectItem--;
            }
            while (currentItem<mzValues.size() && mzValues.get(currentItem)<(currentMz+mzTolerance)){
              if (mzValues.get(currentItem)>=(currentMz-mzTolerance)) sumArea += intensityValues.get(currentItem);
              currentItem++;
            }
            currentItems.put(i, currentItem);
            if (currentItem>(mzValues.size()-1)&&currentItem>0) maxMZs.put(i, mzValues.get(mzValues.size()-1));
          }
          if (sumArea>basePeakArea) basePeakArea = sumArea;
        }
        // alternative method of algorithm - commented because seems to be slower
/****        boolean allFinished = false;
        float currentMz = lowestMz;
        //read out all values and calculate base peak
        while (!allFinished){
//          System.out.println(count1+";"+currentMz);
          float sumArea = 0;
          for (int i=0;i!=mzs.size();i++){
            Vector<Float> mzValues = mzs.get(i);
            Vector<Float> intensityValues = intensities.get(i);
            int currentItem = currentItems.get(i);
            int selectItem = currentItem-1;
            while (selectItem>-1 && (mzValues.get(selectItem)>=(currentMz-mzTolerance))){
              if (mzValues.get(selectItem)<(currentMz+mzTolerance)) sumArea += intensityValues.get(selectItem);
              selectItem--;
            }
            while (currentItem<mzValues.size() && mzValues.get(currentItem)<(currentMz+mzTolerance)){
              if (mzValues.get(currentItem)>=(currentMz-mzTolerance)) sumArea += intensityValues.get(currentItem);
              currentItem++;
            }
            currentItems.put(i, currentItem);
            if (currentItem>(mzValues.size()-1)) maxMZs.put(i, mzValues.get(mzValues.size()-1));
          }
          if (sumArea>basePeakArea) basePeakArea = sumArea;
          currentMz += mzTolerance;
          //should it stop?
          boolean notFinished = false;
          for (int i=0;i!=mzs.size();i++){
            if (currentMz<(maxMZs.get(i)-mzTolerance)){
              notFinished = true;
              break;
            }
          }
          if (!notFinished) allFinished = true;
        }*/
      }else
        //TODO: the implementation for not precursor chrom falses
        throw new CgException("The chrom2 type "+CHROMATOGRAM_HEADER_FILE_MSMS_TYPE_FULL+" is not implemented yet!");
      basePeaks.put(msLevel, basePeakArea);
    }
    return basePeaks;
  }
  
  /**
   * ATTENTION: prepareMSnSpectraCache has to be called before
   * calculates how many percents of the used spectra are covered by found hits
   * @param foundHits identified m/z values
   * @param notCounted hits that originate from another species; these hits should be excluded from the spectrum coverage
   * @param levels for which MSn levels should the base peak values be extracted
   * @param startTime lower time threshold in seconds
   * @param stopTime upper time threshold in seconds
   * @param basePeakCutoff relational value for discarding small peak intensities
   * @return hash table containing the coverages
   * @throws CgException
   */
  public Hashtable<Integer,Float> calculateSpectrumCoverage(Vector<CgProbe> foundHits, Vector<CgProbe> notCounted, Vector<Integer> levels, float startTime, float stopTime, float basePeakCutoff) throws CgException {
    Hashtable<Integer,Float> spectrumCoverage = new Hashtable<Integer,Float>();
    float cutoffMultiplier = 0.0001f;
    if (basePeakCutoff>cutoffMultiplier) cutoffMultiplier = basePeakCutoff;
    for (Integer msLevel : levels){
      float totalIntensity = 0f;
      float coveredIntensity = 0f;
      if (msmsType_!=null && msmsType_.equalsIgnoreCase(CHROMATOGRAM_HEADER_FILE_MSMS_TYPE_PRECURSOR)){
        Hashtable<Integer,String> allSpectra = msnSpectraCache_.get(msLevel);
        Hashtable<Integer,Float> retTimes = msmsRetentionTimes_.get(msLevel);
        Hashtable<Integer,Float> noiseLevels = msnSpectraNoise_.get(msLevel);
//        float lowestMz = Float.MAX_VALUE;
        float highestIntensity = 0f;
        Hashtable<Integer,Vector<Float>> mzs = new Hashtable<Integer,Vector<Float>>();
        Hashtable<Integer,Vector<Float>> intensities = new Hashtable<Integer,Vector<Float>>();
        Hashtable<Integer,Double> retts = new Hashtable<Integer,Double>();
        Hashtable<Integer,Float> noise = new Hashtable<Integer,Float>();
        int count = 0;
        for (int consScanNumber : allSpectra.keySet()){
          double retTime = retTimes.get(consScanNumber);
          if (startTime<=retTime && retTime <= stopTime){
            float noiseThreshold = noiseLevels.get(consScanNumber)*NOISE_CUTOFF_MULTIPLICATOR;
            String spectrum = allSpectra.get(consScanNumber);
            spectrum = spectrum.substring(spectrum.indexOf(" ")+1);
            FloatBuffer buffer = ByteBuffer.wrap(Base64.decode(spectrum)).asFloatBuffer();
            int limit = buffer.limit();
            float mz = 0f;
            float intensity = 0f;
            Vector<Float> mzValues = new Vector<Float>();
            Vector<Float> intensityValues = new Vector<Float>();
            for(int iItem = 0; iItem < limit; iItem++){
              if (iItem%2==0) mz = buffer.get();
              else{
                intensity = buffer.get();
                if (intensity>noiseThreshold){
                  mzValues.add(mz);
                  intensityValues.add(intensity);
                  if (intensity>highestIntensity) highestIntensity = intensity;
                }
              }
            }
            mzs.put(count, mzValues);
            intensities.put(count, intensityValues);
            retts.put(count, retTime);
            noise.put(count, noiseThreshold);
            count++;
          }
        }
        for (int i=0;i!=mzs.size();i++){
          Vector<Float> mzValues = mzs.get(i);
          Vector<Float> intensityValues = intensities.get(i);
          float retTime = retts.get(i).floatValue();
          float noiseThreshold = noise.get(i);
          for (int j=0; j!=mzValues.size(); j++){
            float mz = mzValues.get(j);
            float intensity = intensityValues.get(j);
            if (intensity>(highestIntensity*cutoffMultiplier) &&  intensity>noiseThreshold){
              boolean countIt = true;
              for (CgProbe nc : notCounted){
                if (nc.getMsLevel()==msLevel&&(nc.Mz-nc.LowerMzBand)<=mz && mz<=(nc.Mz+nc.UpperMzBand) && nc.LowerValley<=retTime && retTime<=nc.UpperValley){
                  countIt = false;
                  break;
                }                
              }
              if (!countIt) continue;
              totalIntensity += intensity;
              for (CgProbe hit : foundHits){
                if (hit.getMsLevel()==msLevel&&(hit.Mz-hit.LowerMzBand)<=mz && mz<=(hit.Mz+hit.UpperMzBand) && hit.LowerValley<=retTime && retTime<=hit.UpperValley){
                  coveredIntensity+=intensity;
                  break;
                }
              }
            }
          }
        }
      }else
        //TODO: the implementation for not precursor chrom falses
        throw new CgException("The chrom2 type "+CHROMATOGRAM_HEADER_FILE_MSMS_TYPE_FULL+" is not implemented yet!");
      spectrumCoverage.put(msLevel, (coveredIntensity/totalIntensity));
    }
    return spectrumCoverage;
  }
  
  /**
   * 
   * @return the cached spectra for a certain precursor range
   */
  public Hashtable<Integer,Hashtable<Integer,String>> getMSnSpectraCache(){
    return this.msnSpectraCache_;
  }
  
  /**
   * 
   * @return the cached noise for a certain precursor range
   */
  public Hashtable<Integer,Hashtable<Integer,Float>> getMSnSpectraNoise(){
    return this.msnSpectraNoise_;
  }
  
  /**
   * creates the time points required for interpolation (depending on the smooth range of the chromatogram)
   * @param chromSmoothRange the smooth range of the chromatogram in seconds
   */
  private void interpolateRetentionTimes(float chromSmoothRange){
    Hashtable<Integer,Float> retentionTimesNew = new Hashtable<Integer,Float>();;
    float stepSize = chromSmoothRange/5f;
    float currentRt = this.retentionTimes_.get(0);
    int add = 0;
    while (currentRt<retentionTimes_.get(retentionTimes_.size()-1)){
      retentionTimesNew.put(add, currentRt);
      currentRt += stepSize;
      add++;
    }
    this.retentionTimes_ = retentionTimesNew;
    this.numberOfScans = this.retentionTimes_.size();
  }
  
}