package at.tugraz.genome.maspectras.quantification;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.math.BigDecimal;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Set;
import java.util.Vector;
import java.util.Properties;
import java.util.Hashtable;
import java.io.DataOutputStream;

import at.tugraz.genome.dbutilities.Base64;
import at.tugraz.genome.maspectras.GlobalConstants;
import at.tugraz.genome.maspectras.utils.Calculator;
import at.tugraz.genome.maspectras.utils.StringUtils;
import at.tugraz.genome.util.BioUtilsConstants;
import at.tugraz.genome.voutils.GeneralComparator;

/** 
 * Translates the original raw files into the chrom file for the calculation
 * @author Juergen Hartler
 *
 */
public class RawToChromatogramTranslator implements CgIAddScan
{
  /** My Scans*/
//  private CgScan[]m_scans;
  /** actual Scan Count (used as index) */
  private Hashtable<String,Integer> m_scanCount;
  /** My Header */
//  private CgScanHeader current_header;
  /** The mzXml File Name */
  private String m_fileName;
  /** the lowest possible mz value*/
  private int lowestMz;
  /** the highest possible mz value*/
  private int highestMz;
  /** the name of the chromatography file*/
  private Hashtable<String,String> chromFileName_;
  /** the name of the index file*/
  private Hashtable<String,String> indexFileName_;
  /** the name of the header file*/
  private Hashtable<String,String> headerFileName_;
  /** the name of the retention-time file*/
  private Hashtable<String,String> retentionTimeFileName_;
  /** boundaries to translate the original file in several steps as vector*/
  private Vector<Integer> readerBoundaries_;
  /** boundaries to translate the original file in several steps as lookup hash*/
  private Hashtable<Integer,Integer> newReaderBoundaries_;
  /** the reader*/
  private CgReader m_reader;
  
  private int multiplicationFactorForInt_;
  private int lowestResolution_;
  private int elementsForBatchCalculation_;
  
  private String fileType_; 
  private int numberOfIterations_;
  private boolean msms_;
  private int highestMsLevel_;
  
  private Hashtable<String,Hashtable<Integer,Integer>> msMsExclusionList_;
  private Hashtable<String,Hashtable<Integer,Integer>> numberOfMs2Scans_;
  
  private boolean msmsInSeveralFiles_;
  private int msmsSeveralFilesHighestLevel_;
  
  private Hashtable<Integer,Integer> lowestMzs_;
  private Hashtable<Integer,Integer> highestMzs_;
  private Hashtable<Integer,Hashtable<String,Integer>> scanCounts_;
  
  private Hashtable<String,CgScanHeader> headerHash_;
  private Hashtable<String,CgScan[]> scanHash_;
  private Hashtable<String,Integer> highestMsLevelHash_;
  
  
  private String currentFileName_;
  
  private final static String NO_NAME_PLACEHOLDER = "";
 
  public RawToChromatogramTranslator(String mzXmlPath,String fileType, int numberOfIterations, int multiplicationFactorForInt,
      int lowestResulution)
  {
   this(mzXmlPath,fileType,numberOfIterations);
   multiplicationFactorForInt_ = multiplicationFactorForInt;
   lowestResolution_ = lowestResulution;
   elementsForBatchCalculation_ = multiplicationFactorForInt_/lowestResolution_;
  }
  
  /**
   * constructor for the translation
   * @param mzXmlPath the original file
   * @param fileType the type of the file
   * @param numberOfIterations in how many steps should the file be translated
   */
  public RawToChromatogramTranslator(String mzXmlPath,String fileType, int numberOfIterations)
  {
    multiplicationFactorForInt_ = CgDefines.mzMultiplicationFactorForInt;
    lowestResolution_ = CgDefines.lowestResolution;
    elementsForBatchCalculation_ = CgDefines.elementsForBatchCalculation;
    m_fileName = mzXmlPath;
    fileType_ = fileType;
    numberOfIterations_ = numberOfIterations;
    msms_ = false;
    highestMsLevelHash_ = new Hashtable<String,Integer>();
    initStoreHashes();
    highestMsLevel_ = 1;
  }

  public RawToChromatogramTranslator(String mzXmlPath,String fileType, int numberOfIterations, int multiplicationFactorForInt,
      int lowestResulution, boolean msms)
  {
   this(mzXmlPath,fileType,numberOfIterations,multiplicationFactorForInt,lowestResulution);
   msms_ = msms;
  }

  
  public void AddHeader(CgScanHeader hx) throws CgException
  {
    currentFileName_ = NO_NAME_PLACEHOLDER;
    if (hx.fileName!=null) currentFileName_ = hx.fileName; 
    headerHash_.put(currentFileName_, hx);
    scanHash_.put(currentFileName_, new CgScan[hx.ScanCount]);
    m_scanCount.put(currentFileName_, 0);
  }

  public void AddScan(CgScan sx) throws CgException
  {
    if (!scanHash_.containsKey(currentFileName_)) throw new CgException("m_scans Array not allocated");
    if (scanHash_.get(currentFileName_).length==m_scanCount.get(currentFileName_)) return;
    scanHash_.get(currentFileName_)[m_scanCount.get(currentFileName_)] = sx;
    m_scanCount.put(currentFileName_, m_scanCount.get(currentFileName_)+1);
  }

  public void setStartStopHeader(CgScanHeader hx) throws CgException
  {
    if (!headerHash_.containsKey(currentFileName_)) throw new CgException("No header defined");
    headerHash_.get(currentFileName_).StartTime=hx.StartTime;
    headerHash_.get(currentFileName_).EndTime = hx.EndTime;
  }
  
  private void readHeaderInformation(int msLevel){
    String suffix = "";
    chromFileName_ = new Hashtable<String,String>();
    indexFileName_ = new Hashtable<String,String>();
    headerFileName_ = new Hashtable<String,String>();
    retentionTimeFileName_ = new Hashtable<String,String>();
    if (msLevel>1) suffix = String.valueOf(msLevel);
    m_scanCount = new Hashtable<String,Integer>();
    newReaderBoundaries_ = new Hashtable<Integer,Integer>();
    readerBoundaries_ = new Vector<Integer>();
    msMsExclusionList_ = new Hashtable<String,Hashtable<Integer,Integer>>();
    try
    {
      m_reader = null;
      String justFileName = "";
      //check if several mzXML files exist (several MS/MS traces)
      if (fileType_.equalsIgnoreCase("mzXML")){
        m_reader = new CgMzXmlReader(this, msms_&&!msmsInSeveralFiles_,multiplicationFactorForInt_);
        if (m_fileName.endsWith(".mzXML")) justFileName = m_fileName.substring(0, m_fileName.length()-(".mzXML").length());
        else justFileName = m_fileName; 
      }
      if (fileType_.equalsIgnoreCase("mzData")){
        System.out.println("Taking a mzData File");
        m_reader = new CgMzDataReader(this,multiplicationFactorForInt_);
        if (m_fileName.endsWith(".mzData")) justFileName = m_fileName.substring(0, m_fileName.length()-(".mzData").length());
        else justFileName = m_fileName;
      }
      if (fileType_.equalsIgnoreCase("RAW")){
        System.out.println("Taking a XCalibur RAW File");
        m_reader = new CgXCaliburRawReader(this,multiplicationFactorForInt_);
        if (m_fileName.endsWith(".RAW")) justFileName = m_fileName.substring(0, m_fileName.length()-(".RAW").length());
        else justFileName = m_fileName;
      }
      if (m_reader!=null){
        if (numberOfIterations_>1){
          m_reader.ReadFile(m_fileName+suffix,true);
        }else{
          m_reader.ReadFile(m_fileName+suffix);
        }
        if (this.headerHash_.size()==1){
          String justTheName = StringUtils.getJustFileName(justFileName);
          String chromDirPath = justFileName+".chrom";
          generateChromFileNames(justTheName, chromDirPath, headerHash_.keySet().iterator().next(), suffix);
        } else if (this.headerHash_.size()>1){
          String directory = StringUtils.extractDirName(m_fileName);
          for (String name : headerHash_.keySet()){
            String chromDirPath = directory+File.separator+name+".chrom";
            generateChromFileNames(name, chromDirPath, name, suffix);
          }
        }
        this.highestMz = m_reader.getHighestMz();
        this.lowestMz = m_reader.getLowestMz();
        lowestMzs_.put(msLevel, lowestMz);
        highestMzs_.put(msLevel, highestMz);
        int nrOfLLowResItems = this.highestMz-this.lowestMz;
        for (int i=0; i!=numberOfIterations_-1;i++){
          int lineNumber = ((i+1)*nrOfLLowResItems)/numberOfIterations_;
          this.readerBoundaries_.add(new Integer(lineNumber+this.lowestMz));
          this.newReaderBoundaries_.put(new Integer(lineNumber+this.lowestMz), new Integer(i));
        }
      }else throw new CgException("No readable file format.");
    }
    catch(Exception ex)
    {
      ex.printStackTrace();
    }
    for (String fileName: headerHash_.keySet()){
//      System.out.println("originalFile: "+fileName);
      if (headerHash_.get(fileName).highestMSLevel>highestMsLevel_)
        highestMsLevel_ = headerHash_.get(fileName).highestMSLevel;
      highestMsLevelHash_.put(fileName, headerHash_.get(fileName).highestMSLevel);
      msMsExclusionList_.put(fileName, new Hashtable<Integer,Integer>());
//      System.out.println("StartTime: "+headerHash_.get(fileName).StartTime);
//      System.out.println("EndTime: "+headerHash_.get(fileName).EndTime);
//      System.out.println("ScanCount: "+headerHash_.get(fileName).ScanCount);
//      System.out.println("highestMSLevel: "+headerHash_.get(fileName).highestMSLevel);
//      System.out.println("m_scans.length: "+scanHash_.get(fileName).length);  
    }
//    System.out.println("MScanCount: "+m_scanCount);
//    System.out.println("highestMSLevel: "+highestMsLevel_); 
  }
  
  public void translateToChromatograms() throws CgException{
    msmsInSeveralFiles_ = false;
    msmsSeveralFilesHighestLevel_ = 1;
    lowestMzs_ = new Hashtable<Integer,Integer>();
    highestMzs_  = new Hashtable<Integer,Integer>();
    scanCounts_ = new Hashtable<Integer,Hashtable<String,Integer>>();
    if (msms_){
      int msLevel = 2;
      while ((new File(m_fileName+String.valueOf(msLevel))).exists()){
        msmsInSeveralFiles_ = true;
        msmsSeveralFilesHighestLevel_ = msLevel;
        msLevel++;
      }
      for (int i=msmsSeveralFilesHighestLevel_;i!=0;i--)translateToChromatograms(i); 
    }else{
      translateToChromatograms(1);
    }
    try {
      this.writeHeaderFile();
    }
    catch (IOException e) {
      e.printStackTrace();
      throw new CgException(e.getMessage());
    }

  }
  
  /**
   * starts the translation process
   * @throws CgException
   */
  private void translateToChromatograms(int msLevel) throws CgException{
    this.readHeaderInformation(msLevel);
    float intensity;
    if (this.newReaderBoundaries_.size()>0){
      m_scanCount = new Hashtable<String,Integer>();
    }
    String suffix = "";
    if (msLevel>1) suffix = String.valueOf(msLevel);
    try{
      Hashtable<String,Vector<LinkedHashMap<String,Vector<MsMsScan>>>> msmsScans = new Hashtable<String,Vector<LinkedHashMap<String,Vector<MsMsScan>>>>();
      Hashtable<String,Vector<Hashtable<Integer,Float>>> msmsRetentionTimes = new Hashtable<String,Vector<Hashtable<Integer,Float>>>();
      for (String key : headerHash_.keySet()){
        Vector<Hashtable<Integer,Float>> retsOneMzxml = new Vector<Hashtable<Integer,Float>>();
        for (int i=2; i<=highestMsLevel_;i++) retsOneMzxml.add(new Hashtable<Integer,Float>());
        msmsRetentionTimes.put(key, retsOneMzxml);
      }
      if (this.newReaderBoundaries_.size()>0){
        m_reader.setLowerThreshold(this.lowestMz/multiplicationFactorForInt_);
        if (this.readerBoundaries_!=null&&this.readerBoundaries_.size()>0)
          m_reader.setUpperThreshold((this.readerBoundaries_.get(0))/multiplicationFactorForInt_+CgDefines.overlapAllowed);
        else
          m_reader.setUpperThreshold((this.highestMz/multiplicationFactorForInt_)+1);
        
        System.out.println("Starting iteration 1");
        m_reader.ReadFile(m_fileName+suffix);
      }
      scanCounts_.put(msLevel, m_scanCount);      
      msmsScans = groupTheMsMsScans(msmsRetentionTimes);
      this.writeRetentionTimeFile();
      
      long startTime = System.currentTimeMillis();
      Hashtable<String,BufferedOutputStream> stream = new Hashtable<String,BufferedOutputStream>();
      Hashtable<String,DataOutputStream> streamIndex = new Hashtable<String,DataOutputStream>();
      Hashtable<String,float[][]> intensityValuesSection = new Hashtable<String,float[][]>();
      Hashtable<String,Vector<BufferedOutputStream>> msmsChromStreams = new Hashtable<String,Vector<BufferedOutputStream>>();
      Hashtable<String,Vector<DataOutputStream>> msmsIndexStreams = new Hashtable<String,Vector<DataOutputStream>>();
      Hashtable<String,Hashtable<Integer,Integer>> mzIndizes = new Hashtable<String,Hashtable<Integer,Integer>>();
      Hashtable<String,Hashtable<Integer,Long>> bytesIndices = new Hashtable<String,Hashtable<Integer,Long>>();
      for (String key : headerHash_.keySet()){
        stream.put(key, new BufferedOutputStream(new FileOutputStream(chromFileName_.get(key))));
        streamIndex.put(key,new DataOutputStream(new BufferedOutputStream(new FileOutputStream(indexFileName_.get(key)))));
        System.out.println(chromFileName_.get(key));
        intensityValuesSection.put(key,createCleanIntensityArray(m_scanCount.get(key),elementsForBatchCalculation_));
        // this is for the MSMS
        msmsChromStreams.put(key, new Vector<BufferedOutputStream>());
        msmsIndexStreams.put(key, new Vector<DataOutputStream>());
        mzIndizes.put(key, new Hashtable<Integer,Integer>());
        bytesIndices.put(key, new Hashtable<Integer,Long>());
        if (msms_&&!msmsInSeveralFiles_){
          for (int i=2; i<=highestMsLevel_;i++){
            BufferedOutputStream streamChrom2 = new BufferedOutputStream(new FileOutputStream(chromFileName_.get(key)+String.valueOf(i)));
            msmsChromStreams.get(key).add(streamChrom2);
            DataOutputStream streamIndex2 = new DataOutputStream(new BufferedOutputStream(new FileOutputStream(indexFileName_.get(key)+String.valueOf(i))));
            msmsIndexStreams.get(key).add(streamIndex2);
            mzIndizes.get(key).put(i-2, 0);
            bytesIndices.get(key).put(i-2, 0l);
          }
        }
      }
      writeChrom2(msmsChromStreams,msmsIndexStreams,msmsScans,mzIndizes,bytesIndices);
      Hashtable<String,Long> bytesIndex = new Hashtable<String,Long>();
      for (String key : headerHash_.keySet()) bytesIndex.put(key, 0l);
      for (int i=lowestMz; i<highestMz; i+=lowestResolution_){
        if (this.newReaderBoundaries_.containsKey(new Integer(i))){
          m_scanCount = new Hashtable<String,Integer>();
          int numberInVector = newReaderBoundaries_.get(i);
          System.out.println("Starting iteration "+(numberInVector+2));
          this.initStoreHashes();
          m_reader.setLowerThreshold((i/multiplicationFactorForInt_)-CgDefines.overlapAllowed);
          if (this.readerBoundaries_.size()>(numberInVector+1)){
            m_reader.setUpperThreshold((this.readerBoundaries_.get(numberInVector+1))/multiplicationFactorForInt_+CgDefines.overlapAllowed);
          }else
            m_reader.setUpperThreshold((this.highestMz/multiplicationFactorForInt_)+1);
          m_reader.ReadFile(m_fileName+suffix);
          msmsScans = groupTheMsMsScans(msmsRetentionTimes);
          writeChrom2(msmsChromStreams,msmsIndexStreams,msmsScans,mzIndizes,bytesIndices);
        }
        if (((i-lowestMz)/lowestResolution_) % elementsForBatchCalculation_==0){
          for (String key : headerHash_.keySet()){
            intensityValuesSection.put(key,createCleanIntensityArray(m_scanCount.get(key),elementsForBatchCalculation_));
            for (int j=0; j<m_scanCount.get(key); j++){
               scanHash_.get(key)[j].fillIntensitiyArray(intensityValuesSection.get(key)[j],(Float.parseFloat((String.valueOf(i)))/multiplicationFactorForInt_),(Float.parseFloat((String.valueOf(i+elementsForBatchCalculation_*lowestResolution_)))/multiplicationFactorForInt_),elementsForBatchCalculation_);
            }
          }
        }
        
        for (String key : headerHash_.keySet()){
          Vector<Integer> scanNumbers = new Vector<Integer>();
          Vector<Float> intensities = new Vector<Float>();
          for (int j=0; j<m_scanCount.get(key); j++){
            intensity = intensityValuesSection.get(key)[j][((i-lowestMz)/lowestResolution_) % elementsForBatchCalculation_];

            if (intensity>0){
              scanNumbers.add(new Integer(j));
              intensities.add(new Float(intensity));
            }
          }
          String chromString = "";
          if (scanNumbers.size()>0){
            ByteBuffer buffer = ByteBuffer.allocate(scanNumbers.size() * 2 * 4);
            for (int j=0;j!=scanNumbers.size();j++){
              buffer.putInt(scanNumbers.get(j).intValue());
              buffer.putFloat(intensities.get(j).floatValue());
            }
            chromString = String.valueOf(Base64.encode(buffer.array()));
          }
          byte[] bytesToWrite = (chromString+"\n").getBytes();
          stream.get(key).write(bytesToWrite);
          long bytesIdx = bytesIndex.get(key);
          if (((i-lowestMz)/lowestResolution_) % CgDefines.numberOfEntriesForIndex==0){
            streamIndex.get(key).writeInt((i-lowestMz)/lowestResolution_);
            streamIndex.get(key).writeLong(bytesIdx);
          }
          bytesIdx+=bytesToWrite.length;
          bytesIndex.put(key, bytesIdx);
        }
      }
       numberOfMs2Scans_ = new Hashtable<String,Hashtable<Integer,Integer>>();
      for (String key : headerHash_.keySet()){
        stream.get(key).close();
        stream.get(key).flush();
        streamIndex.get(key).close();
        streamIndex.get(key).flush();
        if (msms_&&!msmsInSeveralFiles_){
          numberOfMs2Scans_.put(key, new Hashtable<Integer,Integer>());
          writeMs2RetentionTimes(msmsRetentionTimes.get(key),numberOfMs2Scans_.get(key),key);
          for (int i=0; i<highestMsLevel_-1;i++){
            BufferedOutputStream streamChrom2 = msmsChromStreams.get(key).get(i);
            streamChrom2.close();
            streamChrom2.flush();
            DataOutputStream streamIndex2 = msmsIndexStreams.get(key).get(i);
            streamIndex2.close();
            streamIndex2.flush();
          }
        }
      }
      System.out.println(((System.currentTimeMillis()-startTime)/1000)+" secs");
    }catch(IOException iox){
      iox.printStackTrace();
    }
  }
  
  /**
   * writes the header file
   */
  private void writeHeaderFile() throws IOException {
    for (String key : this.headerFileName_.keySet()){
      Properties props = new Properties();
      props.put(GlobalConstants.CHROMATOGRAM_HEADER_FILE_MZ_MULTIPLICATION_FACTOR, String.valueOf(multiplicationFactorForInt_));
      props.put(GlobalConstants.CHROMATOGRAM_HEADER_FILE_LOWEST_RESOLUTION,String.valueOf(lowestResolution_));
      props.put(GlobalConstants.CHROMATOGRAM_HEADER_FILE_LOWEST_MZ, String.valueOf(this.lowestMz));
      props.put(GlobalConstants.CHROMATOGRAM_HEADER_FILE_HIGHEST_MZ, String.valueOf(this.highestMz));
      props.put(GlobalConstants.CHROMATOGRAM_HEADER_FILE_NUMBER_SCANS, String.valueOf(m_scanCount.get(key)));   
      props.put(BioUtilsConstants.INDEX_HEADER_FILE_NUMBER_OF_ENTRIES_FOR_INDEX, String.valueOf(CgDefines.numberOfEntriesForIndex));
      props.put(BioUtilsConstants.INDEX_HEADER_FILE_INDEX_FILE, this.indexFileName_.get(key));
      props.put(BioUtilsConstants.INDEX_HEADER_FILE_INDEXED_FILE, this.chromFileName_.get(key));
      props.put(GlobalConstants.CHROMATOGRAM_HEADER_FILE_RETENTION_TIME, this.retentionTimeFileName_.get(key));
      int highestLevel = this.highestMsLevelHash_.get(key);
      if (msmsInSeveralFiles_) highestLevel = msmsSeveralFilesHighestLevel_;
      props.put(GlobalConstants.CHROMATOGRAM_HEADER_FILE_MS_LEVEL, String.valueOf(highestLevel));
      if (this.msms_ && highestLevel>1){
        if (msmsInSeveralFiles_){
          props.put(GlobalConstants.CHROMATOGRAM_HEADER_FILE_MSMS_TYPE, ChromatogramReader.CHROMATOGRAM_HEADER_FILE_MSMS_TYPE_FULL);
        } else
          props.put(GlobalConstants.CHROMATOGRAM_HEADER_FILE_MSMS_TYPE, ChromatogramReader.CHROMATOGRAM_HEADER_FILE_MSMS_TYPE_PRECURSOR);
        for (int i=2; i<=highestLevel;i++){
          props.put(BioUtilsConstants.INDEX_HEADER_FILE_INDEX_FILE+String.valueOf(i), this.indexFileName_.get(key)+String.valueOf(i));
          props.put(BioUtilsConstants.INDEX_HEADER_FILE_INDEXED_FILE+String.valueOf(i), this.chromFileName_.get(key)+String.valueOf(i));
          props.put(GlobalConstants.CHROMATOGRAM_HEADER_FILE_RETENTION_TIME+String.valueOf(i), this.retentionTimeFileName_.get(key)+String.valueOf(i));
          if (msmsInSeveralFiles_){
            props.put(GlobalConstants.CHROMATOGRAM_HEADER_FILE_LOWEST_MZ+String.valueOf(i), String.valueOf(this.lowestMzs_.get(i)));
            props.put(GlobalConstants.CHROMATOGRAM_HEADER_FILE_HIGHEST_MZ+String.valueOf(i), String.valueOf(this.highestMzs_.get(i)));
            props.put(GlobalConstants.CHROMATOGRAM_HEADER_FILE_NUMBER_SCANS+String.valueOf(i), String.valueOf(scanCounts_.get(i).get(key)));   
          }else{
            props.put(GlobalConstants.CHROMATOGRAM_HEADER_FILE_NUMBER_SCANS+String.valueOf(i), String.valueOf(this.numberOfMs2Scans_.get(key).get(i)));          
          }
        }
      }
      FileOutputStream stream = new FileOutputStream(headerFileName_.get(key));
      props.store(stream, "Header");
      stream.close();
    }
  }
  
  /**
   * writes the retention-time file
   * @throws IOException
   */
  private void writeRetentionTimeFile() throws IOException{
    for (String key : retentionTimeFileName_.keySet()){
      DataOutputStream streamRetTime = null;
      try{
        streamRetTime = new DataOutputStream(new BufferedOutputStream(new FileOutputStream(retentionTimeFileName_.get(key))));
        for (int i=0; i<m_scanCount.get(key); i++){
          streamRetTime.writeInt(this.scanHash_.get(key)[i].Num);
          streamRetTime.writeFloat(this.scanHash_.get(key)[i].RetentionTime);
        }
    }catch (IOException iox){
      iox.printStackTrace();
    }finally{
      if (streamRetTime!=null){
        streamRetTime.close();
        streamRetTime.flush();
      }
    }
    }
  }
  
  @SuppressWarnings("unchecked")
  private Hashtable<String,Vector<LinkedHashMap<String,Vector<MsMsScan>>>> groupTheMsMsScans(Hashtable<String,Vector<Hashtable<Integer,Float>>> retentionTimes){
    Hashtable<String,Vector<LinkedHashMap<String,Vector<MsMsScan>>>> msmsScans = new Hashtable<String,Vector<LinkedHashMap<String,Vector<MsMsScan>>>> ();
    for (String key : this.headerHash_.keySet()){
      Vector<LinkedHashMap<String,Vector<MsMsScan>>> scansOfOneMzXML = new Vector<LinkedHashMap<String,Vector<MsMsScan>>>();
      Hashtable<Integer,Integer> exclusionList = msMsExclusionList_.get(key);
      for (int i=2; i<=highestMsLevel_;i++){
        List<MsMsScan> scansOfOneLevel = new ArrayList<MsMsScan>();
        Hashtable<Integer,Float> rets = retentionTimes.get(key).get(i-2);
        for (CgScan msScan : scanHash_.get(key)){
        //this is because there is too much space allocated in the array
          if (msScan!=null){
            for (CgScan subScan : msScan.getFullSubScans()){
              //the msMsExclusionList is necessary because the scans are read overlapping, if there are several iterations
              if (subScan.MsLevel==i && !exclusionList.containsKey(subScan.Num)){
                scansOfOneLevel.add(((MsMsScan)subScan));
                rets.put(subScan.Num, subScan.RetentionTime);
                exclusionList.put(subScan.Num, subScan.Num);
              }
            }
          }
        }
      //new the scans are sorted according to their precursor mass
        Collections.sort(scansOfOneLevel,new GeneralComparator("at.tugraz.genome.maspectras.quantification.MsMsScan", "getPrecursorMz", "java.lang.Float"));
        LinkedHashMap<String,Vector<MsMsScan>> sortedScans = new LinkedHashMap<String,Vector<MsMsScan>>();
        for (MsMsScan scan : scansOfOneLevel){
          Vector<MsMsScan> samePrecursorMass = new Vector<MsMsScan>();
          if (sortedScans.containsKey(scan.getPrecursorMzAsString())) samePrecursorMass = sortedScans.get(scan.getPrecursorMzAsString());
          samePrecursorMass.add(scan);
          if (samePrecursorMass.size()>1) Collections.sort(samePrecursorMass,new GeneralComparator("at.tugraz.genome.maspectras.quantification.MsMsScan", "getNum", "java.lang.Integer"));
          sortedScans.put(scan.getPrecursorMzAsString(), samePrecursorMass);
        }
        scansOfOneMzXML.add(sortedScans);
      }
      msmsScans.put(key, scansOfOneMzXML);
    }
    return msmsScans;
  }
  
  private void writeChrom2(Hashtable<String,Vector<BufferedOutputStream>> msmsChromStreams, Hashtable<String,Vector<DataOutputStream>> msmsIndexStreams, Hashtable<String,Vector<LinkedHashMap<String,Vector<MsMsScan>>>> msmsScans,Hashtable<String,Hashtable<Integer,Integer>> mzIndices,Hashtable<String,Hashtable<Integer,Long>> bytesIndices) throws IOException{
    for (String key:headerHash_.keySet()){
      Hashtable<Integer,Integer> mzIndicesOneMzxml  = mzIndices.get(key);
      Hashtable<Integer,Long> byteIndicesOneMzxml = bytesIndices.get(key);
      for (int i=0;i!=msmsChromStreams.get(key).size();i++){
        BufferedOutputStream stream = msmsChromStreams.get(key).get(i);
        DataOutputStream streamIndex = msmsIndexStreams.get(key).get(i);
        LinkedHashMap<String,Vector<MsMsScan>> scans = msmsScans.get(key).get(i);
        int mzIndex = mzIndices.get(key).get(i);
        long bytesIndex = bytesIndices.get(key).get(i);
//      bytesIndex+=bytesToWrite.length;
      
        for (String scanKey : scans.keySet()){
        // there are the same amount of index entries as in the normal chrom format
        // e.g. if you have a file from 400-1800Da, a multiplicationFactorForInt_=1000 and a lowestResolution_=2;
        // then you would have (1800-400)*(1000/2)=700 000 lines in the chrom file (because the lowest resolution = 0.002Da)
        // if then CgDefines.numberOfEntriesForIndex = 1000, the index entries must be 700000/1000 = 700;
        // if you would use a lowestResolution_=1, the index file must contain 1400 entries.
        // this culculates the difference in lines to the last lowest mass value (in our case 400Da);
        // e.g. if you have a mass value 402.51Da and the settings with lowestResolution_=2 the calculation would be
        // (402.51*1000-400000)/2=1255 line in the hypothetical chrom file; that means this entry would be after the 2nd index entry (count starts with zero)
          int currentResDiffToLowest = (Math.round(Calculator.roundFloat((Float.parseFloat(scanKey))*(float)this.multiplicationFactorForInt_, 0,BigDecimal.ROUND_UP))-lowestMz)/lowestResolution_;
        //if the current entry exceeds the value of the next index entry -> write the index entry and increase the index-value
          while (currentResDiffToLowest>=mzIndex){
            streamIndex.writeInt(mzIndex);
            streamIndex.writeLong(bytesIndex);
            mzIndex+=CgDefines.numberOfEntriesForIndex;
          }
          boolean foundOneScan = false;
          String toWrite = ">"+scanKey+"\n";
          for (MsMsScan scan : scans.get(scanKey)){
            ByteBuffer buffer = ByteBuffer.allocate(scan.Scan.length * 2 * 4);
            for (int j=0; j<scan.Scan.length; j++){
              buffer.putFloat(scan.Scan[j][0]);
              buffer.putFloat(scan.Scan[j][1]);            
            }
            if (scan.Scan.length>0){
              toWrite+=scan.Num+" "+String.valueOf(Base64.encode(buffer.array()))+"\n";
              foundOneScan=true;
            }  
          }
          if (foundOneScan){
            byte[] bytesToWrite = toWrite.getBytes();
            stream.write(bytesToWrite);
            bytesIndex+=bytesToWrite.length;
          }
        }
        byteIndicesOneMzxml.put(i, bytesIndex);
        mzIndicesOneMzxml.put(i, mzIndex);
      }
      mzIndices.put(key, mzIndicesOneMzxml);
      bytesIndices.put(key, byteIndicesOneMzxml);
    }
  }
  
  private void writeMs2RetentionTimes(Vector<Hashtable<Integer,Float>> msmsRetentionTimes, Hashtable<Integer,Integer> numberOfMs2Scans, String fileName)throws IOException{
    for (int i=2; i<=highestMsLevel_;i++){
      Hashtable<Integer,Float> rets = msmsRetentionTimes.get(i-2);
      DataOutputStream streamRetTime = null;
      List<Integer> scans = new ArrayList<Integer>();
      Set<Integer> keys = rets.keySet();
      for (Integer key : keys)scans.add(key);
      Collections.sort(scans);
      try{
        int count = 0;
        streamRetTime = new DataOutputStream(new BufferedOutputStream(new FileOutputStream(this.retentionTimeFileName_.get(fileName)+String.valueOf(i))));
        for (Integer key : scans){
          streamRetTime.writeInt(key);
          streamRetTime.writeFloat(rets.get(key));
          count++;
        }
        numberOfMs2Scans.put(i, count);
      }catch (IOException iox){
        iox.printStackTrace();
      }finally{
        if (streamRetTime!=null){
          streamRetTime.close();
          streamRetTime.flush();
        }
      }
    }
  }
  
  public CgScan getLastBaseScan(){
    for (int i=(m_scanCount.get(currentFileName_)-1); i>-1; i--){
      if (scanHash_.containsKey(currentFileName_) && scanHash_.get(currentFileName_)[i].MsLevel==1)
        return scanHash_.get(currentFileName_)[i];
    }
    return null;
  }
  
  public void addParentFileName(String fileName) throws CgException{
    if (this.currentFileName_.equalsIgnoreCase(NO_NAME_PLACEHOLDER)){
      CgScanHeader header = headerHash_.get(currentFileName_);
      header.fileName = fileName;
      headerHash_.remove(currentFileName_);
      CgScan[] scans = scanHash_.get(currentFileName_);
      scanHash_.remove(currentFileName_);
      Integer count = m_scanCount.get(currentFileName_);
      m_scanCount.remove(currentFileName_);
      currentFileName_ = fileName;
      header.fileName = fileName;
      headerHash_.put(currentFileName_, header);
      scanHash_.put(currentFileName_, scans);
      m_scanCount.put(currentFileName_, count);
    }
  }
  
  private void generateChromFileNames(String fileName, String chromDirPath, String key, String suffix){
    File chromDir = new File(chromDirPath);
    chromDir.mkdir();
    String filePath = chromDir.getAbsolutePath()+File.separator+fileName;
    chromFileName_.put(key,filePath+".chrom"+suffix);
    indexFileName_.put(key,filePath+".idx"+suffix);
    headerFileName_.put(key,filePath+".head");
    retentionTimeFileName_.put(key,filePath+".rtt"+suffix);

  }
  
  private float[][] createCleanIntensityArray(int x, int y){
    float[][] array = new float[x][y];
    return array;
  }
  
  private void initStoreHashes(){
    scanHash_ = new Hashtable<String,CgScan[]>();
    headerHash_ = new Hashtable<String,CgScanHeader>();

  }
  
  public Set<String> getOriginalFileNames(){
    return this.headerHash_.keySet();
  }
}
