package at.tugraz.genome.maspectras.rawfilereader;

import java.util.ArrayList;
import java.util.logging.Level;
import java.util.logging.Logger;


/** This is the main parser implementation for Finnigan XCalibur
 *  Data files version 1.2. Most of the code is also used for
 *  the 1.3 version of the parser.
 */
public class MSBinaryDataV1d4 extends MSBinaryDataV1d2 {

  protected Logger		logger;

  public MSBinaryDataV1d4(byte filedata[]) throws MSBinaryParserException {
    super(filedata);
    logger=Logger.getLogger(getClass().getName());
  }

/** Returns the offset after reading the header.
 */
  protected int readHeader(byte filedata[], int offset)
                           throws MSBinaryParserException {
    int startoffset=offset;

    if(logger==null) logger=Logger.getLogger(getClass().getName());

// seems that header is 16-byte aligned and alignment bits are random
    dumpGarbage(filedata, offset, 8, Level.FINEST); offset+=8;

    headerfields.put("HEADER_USERNAME1",
                 MSBinaryParser.getString(filedata, offset, 25)); offset+=50;
    headerfields.put("HEADER_USERNAME2",
                 MSBinaryParser.getString(filedata, offset, 25)); offset+=50;

// alignmentbytes for two string names
    int ivalue=(-offset)&0xF;
    if(ivalue!=0) {
      dumpGarbage(filedata, offset, ivalue, Level.FINEST);
      offset+=ivalue;
    }

    headerfields.put("HEADER_USERNAME3",
                 MSBinaryParser.getString(filedata, offset, 25)); offset+=50;
    headerfields.put("HEADER_USERNAME4",
                 MSBinaryParser.getString(filedata, offset, 25)); offset+=50;

// always 3,4,5, purpose unknown
    ivalue=MSBinaryParser.getInt(filedata, offset);
    if((ivalue<3)||(ivalue>5)) {
      throw new MSBinaryParserException("Illegal value 0x"+
                                        Integer.toHexString(ivalue));
    }
    logger.finest("Uninterpreted value at 0x"+Integer.toHexString(offset));
    offset+=4;

// 5c02, 5802, a402, e002 found, perhaps offset to header end
    ivalue=MSBinaryParser.getInt(filedata, offset);
    logger.finest("Uninterpreted value at 0x"+Integer.toHexString(offset));
    offset+=4;

// pos 0x10c: always 0
    for(int scanpos=0; scanpos<0x256; scanpos++) {
      ivalue=MSBinaryParser.getShort(filedata, offset);

      int expectval=0;
      switch(scanpos) {
        case 0x222: expectval=1; break; // 528 - 294 -
        
/*
        case 0x542: expectval=0x4024; break;
        case 0x562: expectval=0x3ff0; break;
        case 0x56c: expectval=0x1; break;
        case 0x570: expectval=0x31; break;
*/
        default:
      }

/***      if(ivalue!=expectval) {
        throw new MSBinaryParserException("Illegal value 0x"+
                                          Integer.toHexString(ivalue)+
                                          " at 0x"+Integer.toHexString(offset)+
                                          " scanpos 0x"+
                                          Integer.toHexString(scanpos));
      }*/
      offset+=2;
    }

// read some path string (location of output file when generated)
// normally starts at 0x5b8
    ivalue=MSBinaryParser.getInt(filedata, offset); offset+=4;
    String str=MSBinaryParser.getString(filedata, offset, ivalue);
    logger.log(Level.FINE, "String {0} unused (original file location?)", str);
    offset+=(ivalue<<1);

// some value, alwas 0 in 1.2, 1.3, 0x15 in 1.4 files
    ivalue=MSBinaryParser.getInt(filedata, offset);
    if(ivalue!=0) {
      throw new MSBinaryParserException("Unexpected value 0x"+
                                        Integer.toHexString(ivalue)+" at 0x"+
                                        Integer.toHexString(offset));
    }
    offset+=4;

// read file name string
    ivalue=MSBinaryParser.getInt(filedata, offset); offset+=4;
    str=MSBinaryParser.getString(filedata, offset, ivalue);
    logger.log(Level.FINE, "String {0} unused (original file name?)", str);
    offset+=(ivalue<<1);

// read some path string
    ivalue=MSBinaryParser.getInt(filedata, offset); offset+=4;
    str=MSBinaryParser.getString(filedata, offset, ivalue);
    logger.log(Level.FINE, "String {0} unused (project location?)", str);
    offset+=(ivalue<<1);

// version???? 
    ivalue=MSBinaryParser.getInt(filedata, offset); offset+=4;
    str=MSBinaryParser.getString(filedata, offset, ivalue);
    if(!"1".equals(str)) {
      throw new MSBinaryParserException("Unexpected string "+str+" at 0x"+
                                        Integer.toHexString(offset));
    }
    offset+=(ivalue<<1);

// null string or 0 value??
    ivalue=MSBinaryParser.getInt(filedata, offset); offset+=4;
    if(ivalue!=0) throw new MSBinaryParserException("Unexpected value");

// Text "NotSupported"
    ivalue=MSBinaryParser.getInt(filedata, offset); offset+=4;
    str=MSBinaryParser.getString(filedata, offset, ivalue);
    if(!("Not Supported".equals(str))&&(!"".equals(str))) {
      throw new MSBinaryParserException("Unexpected string "+str+" at 0x"+
                                        Integer.toHexString(offset));
    }
    offset+=(ivalue<<1);

// null string or 0 value??
    ivalue=MSBinaryParser.getInt(filedata, offset); offset+=4;
    if(ivalue!=0) throw new MSBinaryParserException("Unexpected value");

// 0x14 time 0xff
    for(int scanpos=0; scanpos<0x14; scanpos++) {
      if(filedata[offset++]!=-1) throw new MSBinaryParserException("Unexpected value");
    }

    for(int scanpos=0; scanpos<8; scanpos++) {
      logger.fine("Uninterpreted value 0x"+
                  Integer.toHexString(MSBinaryParser.getInt(filedata, offset))+
                  " at 0x"+Integer.toHexString(offset));
      offset+=4;
    }

    msdatalistsectionoffset=MSBinaryParser.getInt(filedata, offset); offset+=4;

    for(int scanpos=0; scanpos<4; scanpos++) {
      logger.fine("Uninterpreted value 0x"+
                  Integer.toHexString(MSBinaryParser.getInt(filedata, offset))+
                  " at 0x"+Integer.toHexString(offset));
      offset+=4;
    }

    scanlogsectionoffset=MSBinaryParser.getInt(filedata, offset); offset+=4;

    for(int scanpos=0; scanpos<3; scanpos++) {
      logger.fine("Uninterpreted value 0x"+
                  Integer.toHexString(MSBinaryParser.getInt(filedata, offset))+
                  " at 0x"+Integer.toHexString(offset));
      offset+=4;
    }

    for(int scanpos=0; scanpos<0x2e8; scanpos++) {
      if(filedata[offset++]!=0) throw new MSBinaryParserException("Unexpected value");
    }


// lets say that the header ends here: file metainfo (date, creator,
// location is read, offsets to next sections are read, so let it be
    return(offset);
  }


/*

  protected int readMainSection(byte filedata[], int offset)
                                throws MSBinaryParserException {
// read strings again
// Text "Study"
    int ivalue=MSBinaryParser.getInt(filedata, offset); offset+=4;
    String str=MSBinaryParser.getString(filedata, offset, ivalue);
    if(!"Study".equals(str)) {
      throw new MSBinaryParserException("Unexpected string "+str+" at 0x"+
                                        Integer.toHexString(offset));
    }
    offset+=(ivalue<<1);

    logger.info("Skipping data from 0x"+Integer.toHexString(offset)+" to 0x"+
                Integer.toHexString(msdatalistsectionoffset));

// msdata seems to reside between these two offsets
    logger.fine("Reading ms data from 0x"+
                Integer.toHexString(msdatalistsectionoffset)+
                " to 0x"+Integer.toHexString(scanlogsectionoffset));

    MSMZDataPoint mspointlist[]=
          readMSDataList(filedata, msdatalistsectionoffset,
                         scanlogsectionoffset-msdatalistsectionoffset);

    offset=scanlogsectionoffset;

// now 3 values follow, purpose unknown
    for(int scanpos=0; scanpos<3; scanpos++) {
      logger.fine("Uninterpreted value 0x"+
                  Integer.toHexString(MSBinaryParser.getInt(filedata, offset))+
                  " at 0x"+Integer.toHexString(offset));
      offset+=4;
    }

    scancount=MSBinaryParser.getInt(filedata, offset);
    logger.finer("Found scancount "+scancount+" at 0x"+
                 Integer.toHexString(offset));
    offset+=4;

    spectracount=MSBinaryParser.getInt(filedata, offset);
    logger.finer("Found spectracount "+spectracount+" at 0x"+
                 Integer.toHexString(offset));
    offset+=4;

// now 2 values follow, purpose unknown
    for(int scanpos=0; scanpos<2; scanpos++) {
      ivalue=MSBinaryParser.getInt(filedata, offset);
      if(ivalue!=0) throw new MSBinaryParserException("Unexpected value nonzero value 0x"+Integer.toHexString(ivalue)+" at 0x"+Integer.toHexString(offset));
      offset+=4;
    }

// seems offset to next section
    strangescaninfosectionoffset=MSBinaryParser.getInt(filedata, offset);
    logger.fine("Uninterpreted offset value 0x"+
                  Integer.toHexString(strangescaninfosectionoffset)+
                  " at 0x"+Integer.toHexString(offset));
    offset+=4;

// this value should be the start of the ms spectra section (again)
    ivalue=MSBinaryParser.getInt(filedata, offset);
    if(ivalue!=msdatalistsectionoffset) {
      throw new MSBinaryParserException("Unexpected value 0x"+
          Integer.toHexString(ivalue)+" at 0x"+Integer.toHexString(offset)+
          ", expected 0x"+Integer.toHexString(msdatalistsectionoffset));
    }
    offset+=4;


    ivalue=MSBinaryParser.getInt(filedata, offset);
    if(ivalue<offset) throw new MSBinaryParserException("Offset error");
    logger.info("Skipping data from 0x"+Integer.toHexString(offset)+" to 0x"+
                Integer.toHexString(ivalue));
    offset=ivalue;

// read a number of device log file entries, they seem to contain
// a timestamp at the beginning, the remaining data is uninterpreted
    for(int samplecnt=0; ; samplecnt++) {
      ivalue=MSBinaryParser.getInt(filedata, offset+0x32);
      if(ivalue!=0x750052) break;
      logger.fine("Reading log entry "+samplecnt+" (0x"+
                  Integer.toHexString(samplecnt)+") at 0x"+
                  Integer.toHexString(offset));

//      logger.fine("Reading log entry "+samplecnt);

      if(logger.isLoggable(Level.FINEST)) {
        dumpGarbage(filedata, offset, 0x13d);
      }
      offset+=0x13d;
    }


* Read the next section. This section contains blocks of 0x48
 * bytes, one block per scan.
 *
    logger.info("Skipping data from 0x"+Integer.toHexString(offset)+" to 0x"+
                Integer.toHexString(strangescaninfosectionoffset));
    offset=strangescaninfosectionoffset;

    msscanlist=new MSScanInfo[scancount];
// the number of scans already read
    int scanreadcounter=0;
// the number of MSMZDataPoints already associated with a scan
    int datapointsassociated=0;
    while(true) {
      ivalue=MSBinaryParser.getInt(filedata, offset+12);
      if(ivalue!=scanreadcounter+1) {
        if(scanreadcounter!=scancount) {
          logger.severe("Found unexpected scanvalue "+ivalue+", expected "+
                        (scanreadcounter+1)+" (at 0x"+Integer.toHexString(offset)+")");
        }
        break;
      }

      logger.fine("Reading sample "+scanreadcounter+" at 0x"+
                  Integer.toHexString(offset));

// +0x0: byte offset in msmz list where this scan has data 
      ivalue=MSBinaryParser.getInt(filedata, offset);
      if(ivalue!=datapointsassociated*8) {
        throw new MSBinaryParserException("MSMZ list read error");
      }

// +0x4: serial, starting from 0
      ivalue=MSBinaryParser.getInt(filedata, offset+4);
      if(ivalue!=scanreadcounter) {
        throw new MSBinaryParserException("Unexpected scannumber in block 0x"+
                                          Integer.toHexString(offset));
      }


// +0x8: some counter, starts with 0, increases to 4, then 0 again ???

// +0xc: scan id
      int scanid=MSBinaryParser.getInt(filedata, offset+0xc);
      if(scanid!=scanreadcounter+1) {
        throw new MSBinaryParserException("Unexpected scanid in block 0x"+
                                          Integer.toHexString(offset));
      }

// +0x10: some fixed value
      ivalue=MSBinaryParser.getInt(filedata, offset+0x10);
      if(ivalue!=0xf) {
        throw new MSBinaryParserException("Unexpected value 0x"+
                   Integer.toHexString(ivalue)+" at 0x"+
                   Integer.toHexString(offset+4));
      }

// +0x14: number of samples in scan
      ivalue=MSBinaryParser.getInt(filedata, offset+0x14);

      MSMZDataPoint hitbuffer[]=new MSMZDataPoint[ivalue];
      System.arraycopy(mspointlist, datapointsassociated,
                       hitbuffer, 0, ivalue);
      datapointsassociated+=ivalue;

      msscanlist[scanreadcounter]=new MSScanInfo(
               scanreadcounter,
               scanid,
               hitbuffer);

// dump all data
      if(logger.isLoggable(Level.FINEST)) {
        dumpGarbage(filedata, offset, 0x48);
      }
      offset+=0x48;
      scanreadcounter++;
    }
    if(scanreadcounter!=scancount) {
      throw new MSBinaryParserException("Scan info read error, expected "+
                                        scancount+", got "+scanreadcounter);
    }
    if(datapointsassociated!=mspointlist.length) {
      throw new MSBinaryParserException("Failed to associated mspoints with scans, only "+datapointsassociated+" of "+mspointlist.length+" used");
    }
    logger.fine("Offset after reading of scan table 0x"+
                Integer.toHexString(offset));

// now there is some unknown counter, seems to be number of scans
    int unknownscancounter=MSBinaryParser.getInt(filedata, offset);
    if(unknownscancounter!=scancount) {
      throw new MSBinaryParserException("Unexpected scan value 0x"+
                          Integer.toHexString(unknownscancounter)+" at 0x"+
                          Integer.toHexString(offset));
    }
    offset+=4;


// next section, blocks with very low informational content,
// blocks seem to have different sizes


    int blockcount=0;
    while(true) {
// +0x0: value should always be the same (signature?)
      ivalue=MSBinaryParser.getInt(filedata, offset);
      if(ivalue!=0x1020001) {
        logger.fine("Unexpected value 0x"+Integer.toHexString(ivalue)+" at 0x"+Integer.toHexString(offset)+", terminating block recognition");
        break;
      }

// +0x4: always 1
      ivalue=MSBinaryParser.getShort(filedata, offset+4);
      if(ivalue!=1) {
        throw new MSBinaryParserException("Unexpected value 0x"+
                    Integer.toHexString(ivalue)+
                    " in block "+blockcount+" at 0x"+
                    Integer.toHexString(offset)+", offset 0x4");
      }

// +0x6: very few values allowed
// 0x1: found in 1.3-file, first block only: block size reduced to 0x4c
//     in 1.3 files always followed by 0xa000102
//     also in 1.2-file, where very common, blocksize reduced to 0x44
//     but here followed by 0xa000101
// 0x2: found in 1.3-file, when blocksize 0x6c (normal), always
//     followed by 0xa010102
//     Also found in 1.2-file, where followed by 0xa010101
      int blocklength=0x44;
      int blocktype=MSBinaryParser.getShort(filedata, offset+6);
      if(blocktype==0x1) {
      } else if(blocktype==0x2) {
        blocklength+=0x20;
      } else {
        throw new MSBinaryParserException("Unexpected block type 0x"+
                            Integer.toHexString(blocktype)+" at 0x"+
                            Integer.toHexString(offset)+", block offset 0x6");
      }

// +0x8:
      ivalue=MSBinaryParser.getShort(filedata, offset+8);
      if(ivalue==0x101) { // normal in 1.2files
      } else if(ivalue==0x102) {
        blocklength+=0x8;
      } else {
        throw new MSBinaryParserException("Unexpected value 0x"+
                            Integer.toHexString(ivalue)+" at 0x"+
                            Integer.toHexString(offset)+
                            ", block offset 0x8");
      }


// +0xa: In 1.2 files always 0xa00 when blocktype 1,
//       0xa01 when blocktype 2
      ivalue=MSBinaryParser.getShort(filedata, offset+0xa);
      if(((ivalue&0xFF00)!=0xa00)||((ivalue&0xFF)!=blocktype-1)) {
        throw new MSBinaryParserException("Unexpected value 0x"+
                            Integer.toHexString(ivalue)+" at 0x"+
                            Integer.toHexString(offset)+
                            ", block offset 0xa");
      }


      if(logger.isLoggable(Level.FINEST)) {
        logger.finest("Uninterpreted block 0x"+Integer.toHexString(blockcount)+
                      " ("+blockcount+"):");
        dumpGarbage(filedata, offset, blocklength);
      }

      blockcount++;
      offset+=blocklength;
    }

    logger.fine("Read 0x"+Integer.toHexString(blockcount)+" ("+blockcount+
                ") blocks, offset now 0x"+Integer.toHexString(offset));

    if(blockcount!=scancount) {
      throw new MSBinaryParserException("Unexpected blockcount value 0x"+
                          Integer.toHexString(blockcount)+" expected 0x"+
                          Integer.toHexString(scancount));
    }


// next section
// 1 spacer/alignment/unknown byte
    if(filedata[offset]!=0) {
      throw new MSBinaryParserException("Unexpected value at 0x"+
          Integer.toHexString(offset));
    }
    offset++;


    blockcount=0;
    while(offset!=filedata.length) {
      ivalue=MSBinaryParser.getShort(filedata, offset);
      if((ivalue!=0x2)&&(ivalue!=0x3)) {
        logger.fine("Unexpected value 0x"+Integer.toHexString(ivalue)+" at 0x"+Integer.toHexString(offset)+", terminating block recognition");
        break;
      }

      if(logger.isLoggable(Level.FINEST)) {
        logger.finest("Uninterpreted block 0x"+Integer.toHexString(blockcount)+
                      " ("+blockcount+"):");
        dumpGarbage(filedata, offset, 0x1b);
      }

      blockcount++;
      offset+=0x1b;
    }

    logger.fine("Read 0x"+Integer.toHexString(blockcount)+" ("+blockcount+
                ") blocks, offset now 0x"+Integer.toHexString(offset));

    if(blockcount!=scancount) {
      throw new MSBinaryParserException("Unexpected blockcount value 0x"+
                          Integer.toHexString(blockcount+1)+" expected 0x"+
                          Integer.toHexString(scancount));
    }

// 3 null-bytes, then end
    ivalue=MSBinaryParser.getInt(filedata, offset-1);
    if((ivalue>>8)!=0) {
      throw new MSBinaryParserException("Unexpected values at 0x"+Integer.toHexString(offset)+", expected 3 null-bytes");
    }
    offset+=3;



    if(offset!=filedata.length) {
      logger.info("Skipping data from 0x"+Integer.toHexString(offset)+" to 0x"+
                  Integer.toHexString(filedata.length));
    }
  }




** Read the hitlists for all scans. This list is then parted
 *  into the different scans when reading the MSScanInfo
 *  section.
 *
  private MSMZDataPoint[] readMSDataList(byte filedata[],
                                         int offset, int length) {
    if((offset<0)||(length<0)||((length&0x7)!=0)||
       (offset+length>filedata.length)) {
      throw new IllegalArgumentException("Invalid data range");
    }

    MSMZDataPoint pointlist[]=new MSMZDataPoint[length>>3];
    int pointcnt=0;
    int endoffset=offset+length;
    for(; offset<endoffset; offset+=8) {
      int ivalue=MSBinaryParser.getInt(filedata, offset);

// the first byte is the number of 3-bit shifts to apply to
// the detectorhitcnt value
      int detcntshift=(ivalue&0xFF);
      if(detcntshift>3) {
        logger.info("Rare value 0x"+Integer.toHexString(ivalue)+" at 0x"+
                    Integer.toHexString(offset)+" check it");
      }
      long detcnt=(ivalue>>8)&0xFFFFFF;
      while(detcntshift!=0) {
        detcnt<<=3;
        detcntshift--;
      }

      ivalue=MSBinaryParser.getInt(filedata, offset+4);
      double massvalue=((double)(ivalue&0xFFFF))+
                       (((double)((ivalue>>16)&0xFFFF))/65536.0);
      pointlist[pointcnt++]=new MSMZDataPoint(detcnt, massvalue);
    }
    return(pointlist);
  }


  private void dumpGarbage(byte data[], int offset, int length) {
    if(logger.isLoggable(Level.FINEST)) {
      StringBuffer sbuf=new StringBuffer();
      sbuf.append("Putative garbage at 0x").
           append(Integer.toHexString(offset)).
           append(" ignored:");
      for(int pos=0; pos<length; pos++) {
        if((pos&0xF)==0) sbuf.append('\n');
        else sbuf.append(' ');
        byte b=data[offset+pos];
        sbuf.append(Character.forDigit((b&0xF0)>>4, 16)).
             append(Character.forDigit(b&0xF, 16));
      }
      logger.finest(sbuf.toString());
    }
  }
*/

}
