package at.tugraz.genome.util;

import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.LineNumberReader;
import java.io.RandomAccessFile;
import java.util.Hashtable;
import java.util.Properties;
import java.util.Vector;

import at.tugraz.genome.util.index.IndexFileException;
import at.tugraz.genome.util.index.IndexedLineNumberHeaderReader;

/** class to read and/or to index a file according to the lines*/
public class IndexedLineNumberReader
{
  /** file path of the header file*/
  protected String headerFilePath_;
  /** file path of the index file*/
  protected String indexFilePath_;
  /** file path of the data file (to be indexed)*/
  protected String dataFilePath_;
  /** the number of lines for one index entry*/
  protected int numberOfLinesForIndex_;
  /** the total amount of lines in the data file*/
  protected int numberOfLines_;
  /** should the index file be hold in the cache*/
  private boolean cacheIndexFile_;
  /** cachedLineInMemore */
  protected Hashtable<Integer,Integer> cachedLine_;
  /** indexFileInMemory */
  protected Hashtable<Integer,Long> cachedIndex_;
  
  /**
   * The reader can be used in 2 fashions:
   * 1. to read the lines of an indexed file
   * 2. to index a file
   * for the first case only the headerFilePath is necessary, if no indexFilePath and/or dataFilePath is defined the values
   * are read from the header file
   * for the second case only the dataFilePath is necessary, if no headerFilePath and/or indexFilePath is defined the postfix
   * of the filename is cut and ".head" or ".idx" is added correspondingly
   * @param headerFilePath path to the header file
   * @param indexFilePath path to the index file
   * @param dataFilePath path to the data file
   * @param cacheIndexFile should the index file be hold in the cache
   * @throws IndexFileException
   */
  public IndexedLineNumberReader(String headerFilePath, String indexFilePath, String dataFilePath, boolean cacheIndexFile) {
    this.headerFilePath_ = headerFilePath;
    this.indexFilePath_ = indexFilePath;
    this.dataFilePath_ = dataFilePath;
    this.numberOfLinesForIndex_ = 0;
    this.numberOfLines_ = 0;  
    this.cacheIndexFile_ = cacheIndexFile;

  }
  
  
  
  /**
   * this constructor should be used only when a file has to be indexed
   * @param dataFilePath path to the data file
   * @param numberOfLinesForIndex the amount of lines for one index entry
   * @param cacheIndexFile should the index file be hold in the cache
   */
  public IndexedLineNumberReader(String dataFilePath, int numberOfLinesForIndex){
    this(null,null,dataFilePath, false);
    this.numberOfLinesForIndex_ = numberOfLinesForIndex;
  }
  
  /**
   * this constructor should only be used when from an already indexed file should be read
   * @param headerFilePath the path to the header file
   * @param cacheIndexFile should the index file be hold in the cache
   */
  public IndexedLineNumberReader(String headerFilePath, boolean cacheIndexFile){
    this(headerFilePath,null,null, cacheIndexFile);
  }

  /**
   * starts to read at a specified line, and stops at a specific line
   * @param from line to start (included)
   * @param to line to stop (excluded)
   * @throws IndexFileException
   */
  public String[] readLines(int from, int to)throws IndexFileException{
    Vector<String> readFileLines = new Vector<String>();
    if (this.numberOfLinesForIndex_==0) this.readHeader();
    if (this.indexFilePath_==null || this.indexFilePath_.length()<1)
      throw new IndexFileException("If you want to read from a indexed file the index file must be defined");
    if (this.dataFilePath_==null || this.dataFilePath_.length()<1)
      throw new IndexFileException("If you want to read from a indexed file the data file must be defined");
    if (this.cacheIndexFile_&&(this.cachedIndex_==null||this.cachedIndex_.size()<1)){
      cachedLine_ = new Hashtable<Integer,Integer>();
      cachedIndex_ = new Hashtable<Integer,Long>();
      this.readIndexFileWhole(indexFilePath_,cachedLine_,cachedIndex_);
    }
    if (from<this.numberOfLines_){
      int indexNumber = getIndexFileNumber(from,numberOfLinesForIndex_);
      int currentLine = 0;
      long bytesToSkip = 0;
      if (this.cacheIndexFile_){
        currentLine = this.cachedLine_.get(indexNumber);
        bytesToSkip = this.cachedIndex_.get(indexNumber);
      }else{
        try{
          RandomAccessFile stream = new RandomAccessFile(indexFilePath_,"r");
          stream.seek((4+8)*indexNumber);
          currentLine = stream.readInt();
          bytesToSkip = stream.readLong();
          stream.close();
        }
        catch (IOException e) {
          throw new IndexFileException(e.getMessage());
        } 
      }
      RandomAccessFile reader = null; 
      BufferedReader lreader = null;
      String line = "";
      try {
        reader = new RandomAccessFile(dataFilePath_,"r");
        reader.seek(bytesToSkip);
        lreader = new BufferedReader(new FileReader(reader.getFD()));
        while (to>currentLine&&(numberOfLines_)>currentLine/*line!=null*/){
          line = lreader.readLine();
          if (from<=currentLine){
            readFileLines.add(line);
          }
          currentLine ++;
        }
      }
      catch (IOException e) {
        throw new IndexFileException(e.getMessage());
      }finally{
        try{
          if (lreader!=null)       
            if (reader!=null)
              reader.close();
        }catch(IOException e){};
      }
    }
    String[] lines = new String[readFileLines.size()];
    for (int i=0; i!=readFileLines.size();i++){
      lines[i] = readFileLines.get(i);
    }
    return lines;
  }
  
  /**
   * starts the indexing of the data file
   * @throws IndexFileException
   */
  public void createIndexFile()throws IndexFileException{
    if (this.dataFilePath_==null || this.dataFilePath_.length()<1)
      throw new IndexFileException("If you want to create an index file the data path must be set");
    String nameWithoutPostfix = this.removePostFixIfAny(this.dataFilePath_);
    if (this.indexFilePath_==null || this.indexFilePath_.length()<1){
      this.indexFilePath_ = nameWithoutPostfix+".idx";
    }
    if (this.headerFilePath_==null || this.headerFilePath_.length()<1){
      this.headerFilePath_ = nameWithoutPostfix+".head";
    }
    LineNumberReader reader = null;
    DataOutputStream streamIndex = null; 
    try {
      long bytesIndex = 0;
      int lineNumber = 0;
      String line;
      reader = new LineNumberReader(new FileReader(dataFilePath_));
      streamIndex = new DataOutputStream(new BufferedOutputStream(new FileOutputStream(indexFilePath_)));
      while ((line=reader.readLine()) != null) {
        lineNumber = reader.getLineNumber()-1;
        if (lineNumber%this.numberOfLinesForIndex_==0){
          streamIndex.writeInt(lineNumber);
          streamIndex.writeLong(bytesIndex);
        }
        bytesIndex += (line+"\n").length();
 
      }
      if (reader.getLineNumber()>0) lineNumber++;
      this.writeHeaderFile(lineNumber);
    }
    catch (IOException e) {
      throw new IndexFileException(e.getMessage());
    }finally{
      try{
        if (reader!=null)
          reader.close();
        if (streamIndex!=null){
          streamIndex.close();
          streamIndex.flush();
        }
      }catch(IOException e){};
    }

    
  }
  
  /**
   * initializes the reading of the header file
   * @throws IndexFileException
   */
  protected void readHeader() throws IndexFileException{
    if (this.headerFilePath_==null||this.headerFilePath_.length()<1) 
      throw new IndexFileException("If you want to read from a indexed file the header file must be defined");
    IndexedLineNumberHeaderReader headerReader = new IndexedLineNumberHeaderReader(this.headerFilePath_);
    headerReader.readFile();
    this.numberOfLinesForIndex_ = headerReader.getNumberOfLinesForIndex();
    if (this.indexFilePath_==null || this.indexFilePath_.length()<1)
      this.indexFilePath_ = headerReader.getIndexFilePath();
    if (this.dataFilePath_==null || this.dataFilePath_.length()<1)
      this.dataFilePath_ = headerReader.getIndexedFilePath();
    numberOfLines_ = headerReader.getNumberOfLinesTotal();
  }
  
  /**
   * creates a header file
   * @param lineNumber total amount of lines from the data file
   * @throws IOException
   */
  private void writeHeaderFile(int lineNumber) throws IOException {
    Properties props = new Properties();
    props.put(BioUtilsConstants.INDEX_HEADER_FILE_INDEX_FILE, this.indexFilePath_);
    props.put(BioUtilsConstants.INDEX_HEADER_FILE_INDEXED_FILE,this.dataFilePath_);
    props.put(BioUtilsConstants.INDEX_HEADER_FILE_NUMBER_OF_ENTRIES_FOR_INDEX, String.valueOf(this.numberOfLinesForIndex_));
    props.put(BioUtilsConstants.INDEX_HEADER_FILE_NUMBER_OF_LINES, String.valueOf(lineNumber));
    FileOutputStream stream = new FileOutputStream(headerFilePath_);
    props.store(stream, "Header");
    stream.close();
    this.numberOfLines_ = lineNumber;
  }
  
  /**
   * sets the path to the data file manually (better use constructor)
   * @param dataFilePath_ path to the data
   */
  public void setDataFilePath_(String dataFilePath_)
  {
    this.dataFilePath_ = dataFilePath_;
  }

  /**
   * sets the path to the header file manually (better use constructor)
   * @param headerFilePath_ path to the header file
   */
  public void setHeaderFilePath_(String headerFilePath_)
  {
    this.headerFilePath_ = headerFilePath_;
  }

  /**
   * sets the path to the index file manually (better use constructor)
   * @param indexFilePath_ path to the index file
   */
  public void setIndexFilePath_(String indexFilePath_)
  {
    this.indexFilePath_ = indexFilePath_;
  }

  /**
   * sets the number of lines for an index entry manually (better use constructor)
   * @param numberOfLinesForIndex_ number of lines for an index entry
   */
  public void setNumberOfLinesForIndex_(int numberOfLinesForIndex_)
  {
    this.numberOfLinesForIndex_ = numberOfLinesForIndex_;
  }
  
  protected String removePostFixIfAny(String name){
    String nameWithoutPostfix = name;
    if (name.indexOf(".")!=-1){
      nameWithoutPostfix = nameWithoutPostfix.substring(0,nameWithoutPostfix.lastIndexOf("."));
    }
    return nameWithoutPostfix;
  }
  
  protected void readIndexFileWhole(String indexFilePath, Hashtable<Integer,Integer> cachedLine, Hashtable<Integer,Long> cachedIndex) throws IndexFileException{
    if (indexFilePath==null || indexFilePath.length()<1)
      throw new IndexFileException("If you want to read from a indexed file the index file must be defined");
    DataInputStream inStream = null;
    try{
      inStream = new DataInputStream(new FileInputStream(indexFilePath));
      int count = 0;
      while (inStream.available()>0){
        int currentLine = inStream.readInt();
        long bytesToSkip = inStream.readLong();
        cachedLine.put(new Integer(count), new Integer(currentLine));
        cachedIndex.put(new Integer(count), new Long(bytesToSkip));
        count++;
      }
    }catch(IOException iox){
      throw new IndexFileException(iox);
    }finally{
      if (inStream!=null)
        try {inStream.close();}catch (IOException iox){throw new IndexFileException(iox);}
    }
  }
  
  protected int getIndexFileNumber(int from, int numberOfLinesForIndex){
    int indexNumber = 0;
    for (int i=0;i!=from; i+=numberOfLinesForIndex){
      if (i<from) indexNumber++;
      else break;
    }
    if (indexNumber>0&&from%numberOfLinesForIndex!=0) indexNumber--;
    return indexNumber;
  }
}
