/*
 * MainProgram off MPI-Swarm
 * (C) 2002 Gernot Stocker gernot.stocker@tugraz.at
 *
 * Released under GPL
 */
     
#include <stdio.h>
#include <unistd.h>
#include <mpi.h>
#include <string.h>
#include <getopt.h>
#include <sys/types.h>

#include "dirutil.h"
#include "stringutil.h"
#include "swarm.h"
#include "systemcall.h"

char command[CMD_LENGTH+1];
char input_filename[FILENAME_LENGTH+1];
char input_path[FILENAME_LENGTH+1];
char output_path[FILENAME_LENGTH+1];

void usage()
{
  printf("\nUsage: swarm -i input_directory -o output_directory -c command_to_execute\n");
  printf("\t-i input_directory ... directory from which the swarmleader gets\n");
  printf("\t\tall files, replaces the %s string in the run command\n",INPUTFILE_STRING);
  printf("\t\twith the filename of one of the files in the inputdirectory\n");
  printf("\t\tand gives the completed command to one node in the cluster\n"); 
  printf("\t\tfor execution.\n");
  printf("\t-o output_directory ... directory in which to store the output\n"); 
  printf("\t\tof the executed command. As a filename it is taken the \n");
  printf("\t\t%s string extended with a job unique worker id.\n",INPUTFILE_STRING);
  printf("\t-c command_to_run ... command to execute with one of the inputfile\n");
  printf("\t\tusing placeholders %s and %s for\n\t\tthe files.",INPUTFILE_STRING, OUTPUTFILE_STRING);
  printf("\n\nE.G.: \n  mpirun -np 24 swarm -i \"/etc\" -o \"/tmp\" -c \"ls -l %s > %s\"\n\n", INPUTFILE_STRING, OUTPUTFILE_STRING);
  fflush(stderr);
  fflush(stdout);
}

void master_process(int number_of_processes, int number_of_actual_process)
{
    directory_handle_ptr input_dir=NULL;
    char *actual_filename;
    int   assigned_jobs=0;
    int   active_jobs=0;
    int   return_code=-1;
    char  poison[POISON_LENGTH];
    MPI_Status msg_status;
#ifdef DEBUG
    FILE  *logfile;
    logfile=fopen("/tmp/swarm.log", "w+");
#endif    

    /*
     * Master Setup Phase
     */
    snprintf(poison,POISON_LENGTH,"DIE");
    fprintf(stdout, "Info: %d Processors are in use\n", number_of_processes - 1);    
    
    input_dir=new_directory_handle(input_path);

    /*
     * Distribute the command to all Workers
     */    
    MPI_Bcast(command, CMD_LENGTH,MPI_CHAR,MASTER_PROCESS,MPI_COMM_WORLD);    
    /*
     * Distribute the inputpath to all Workers
     */
        
    MPI_Bcast(input_path, FILENAME_LENGTH,MPI_CHAR,MASTER_PROCESS,MPI_COMM_WORLD);    
    /*
     * Distribute the outputpath to all Workers
     */    
    MPI_Bcast(output_path, FILENAME_LENGTH,MPI_CHAR,MASTER_PROCESS,MPI_COMM_WORLD);    
    actual_filename=get_next_filename(input_dir, RETURN_STRING_PTR_MODE);
    if (actual_filename!=NULL) snprintf(input_filename, FILENAME_LENGTH,"%s", actual_filename);

    while (actual_filename!=NULL )
    {
      if (active_jobs < (number_of_processes - 1))
      {
        /*
         * Every worker should get the first job immediately 
         */
        assigned_jobs++;
        active_jobs++;
#ifdef DEBUG
       fprintf(logfile, "PROCESS (%d) (%d): %s \n",number_of_actual_process, getpid(), input_filename);
       fflush(logfile);
#endif
        MPI_Send(input_filename, FILENAME_LENGTH, MPI_CHAR, assigned_jobs, assigned_jobs, MPI_COMM_WORLD);
      }
      else
      {
        /*
	 * Wait for the return of a Processor
	 */
        MPI_Recv(&return_code, 1, MPI_INT, MPI_ANY_SOURCE, MPI_ANY_TAG, MPI_COMM_WORLD, &msg_status);
	active_jobs--;
        /*
	 * Assign a new task to the returned Processor
	 */	
#ifdef DEBUG
       fprintf(logfile, "PREV-RETURN:(%d)\n PROCESS (%d): %s (%d)(%d)(%d)\n",return_code,number_of_actual_process, input_filename,strlen(actual_filename), msg_status.MPI_SOURCE, msg_status.MPI_TAG);
       fflush(logfile);
#endif
	MPI_Send(input_filename, FILENAME_LENGTH, MPI_CHAR, msg_status.MPI_SOURCE, msg_status.MPI_TAG, MPI_COMM_WORLD);
#ifdef DEBUG
       fprintf(logfile, "SENT (%d): %s (%d)(%d)\n",number_of_actual_process, input_filename, msg_status.MPI_SOURCE, msg_status.MPI_TAG);
       fflush(logfile);
#endif
	
	active_jobs++;
      }
      actual_filename=get_next_filename(input_dir, RETURN_STRING_PTR_MODE);
      if (actual_filename!=NULL) snprintf(input_filename, FILENAME_LENGTH,"%s", actual_filename);
    }
    /*
     * Master has to collect results and reasign
     */
    while (active_jobs > 0)
    {
        /*
	 * Wait for the return of a Processor
	 */
        MPI_Recv(&return_code, 1, MPI_INT, MPI_ANY_SOURCE, MPI_ANY_TAG, MPI_COMM_WORLD, &msg_status);
	active_jobs--;
        /*
	 * Finish the Processor
	 */
        MPI_Send(poison, POISON_LENGTH, MPI_CHAR, msg_status.MPI_SOURCE, END_SIGNAL_TAG, MPI_COMM_WORLD); 
    } 
     
    /*
     * Master has to quit the unused Processes as well if they exist
     */
    while ( assigned_jobs < (number_of_processes - 1))
    {
      MPI_Send(poison, POISON_LENGTH, MPI_CHAR, assigned_jobs+1, END_SIGNAL_TAG, MPI_COMM_WORLD); 
      assigned_jobs++;
    }
    
    /*
     * Cleanup of Masterprocess
     */
    delete_directory_handle(input_dir);    
#ifdef DEBUG
       fflush(logfile);
       fclose(logfile);
#endif
    printf("Jobs completed!\n");
    fflush(stdout);
}

void  worker_process(int number_of_processes, int number_of_actual_process)
{
    MPI_Status msg_status;
    char *exec_command, full_path[2*FILENAME_LENGTH+1];
    int return_code=0;
#ifdef DEBUG
    FILE  *logfile;
    logfile=fopen("/tmp/swarm.log", "w+");
#endif    
    /*
     * Worker Setup Phase
     */ 
    /*
     * Distribute the actual command to execute
     */
    MPI_Bcast(command, CMD_LENGTH,MPI_CHAR,MASTER_PROCESS,MPI_COMM_WORLD);
    /*
     * Distribute the inputpath to all Workers
     */    
    MPI_Bcast(input_path, FILENAME_LENGTH,MPI_CHAR,MASTER_PROCESS,MPI_COMM_WORLD);    
    /*
     * Distribute the outputpath to all Workers
     */    
    MPI_Bcast(output_path, FILENAME_LENGTH,MPI_CHAR,MASTER_PROCESS,MPI_COMM_WORLD);    
    /*
     * Get the Parameter for the  Command
     */

    MPI_Recv(input_filename, FILENAME_LENGTH, MPI_CHAR, MASTER_PROCESS, MPI_ANY_TAG, MPI_COMM_WORLD, &msg_status);    

        
    while (msg_status.MPI_TAG != END_SIGNAL_TAG)
    {
       exec_command=strdup(command);

#ifdef DEBUG
       fprintf(logfile, "PROCESS (%d): %s \n",number_of_actual_process, full_path);
       fflush(logfile);
       printf("%d Processing: %s with (%s)\n", number_of_actual_process,full_path , exec_command);
#endif
      snprintf(full_path, 2*FILENAME_LENGTH+1,"%s/%s", input_path,input_filename );
      exec_command=str_replace(exec_command, INPUTFILE_STRING, full_path); 

      snprintf(full_path, 2*FILENAME_LENGTH+1,"%s/%s_%d", output_path,input_filename,number_of_actual_process);
      exec_command=str_replace(exec_command, OUTPUTFILE_STRING, full_path); 
      // return_code=system(exec_command);
      return_code=system_call (exec_command);
      free(exec_command);
#ifdef DEBUG
      printf("%d Finished:(%s) (%s)\n", number_of_actual_process, full_path, exec_command);
      fflush(stdout);
#endif
      /*
       * Return the Exit code of the SystemCall
       */
      MPI_Send(&return_code, 1, MPI_INT, MASTER_PROCESS, msg_status.MPI_TAG, MPI_COMM_WORLD);
      /*
       * Wait for new tasks
       */
      MPI_Recv(input_filename, FILENAME_LENGTH, MPI_CHAR, MASTER_PROCESS, MPI_ANY_TAG, MPI_COMM_WORLD, &msg_status);
    }
#ifdef DEBUG
       fflush(logfile);
       fclose(logfile);
#endif
}


void kill_all_workers(int number_of_processes)
{
  char *poison="DIE";
  int assigned_jobs=0;  

  MPI_Bcast(command, CMD_LENGTH,MPI_CHAR,MASTER_PROCESS,MPI_COMM_WORLD);
  MPI_Bcast(input_path, FILENAME_LENGTH,MPI_CHAR,MASTER_PROCESS,MPI_COMM_WORLD);    
  MPI_Bcast(output_path, FILENAME_LENGTH,MPI_CHAR,MASTER_PROCESS,MPI_COMM_WORLD);    

  for ( assigned_jobs=1; assigned_jobs < number_of_processes; assigned_jobs++ )
  {
    fflush(stdout);
    MPI_Send(poison, POISON_LENGTH, MPI_CHAR, assigned_jobs, END_SIGNAL_TAG, MPI_COMM_WORLD); 
  }
}


int main(int argc, char **argv)
{
  int number_of_processes=-1, number_of_actual_process=-1, c, option_index = 0, options[3];
  directory_handle_ptr ouput_dir_handle=NULL;
  extern char *optarg;
  extern int optind, opterr, optopt;
  static struct option long_options[] =
             {
               {"command", 1, 0, 'c'},
               {"inputdir", 0, 0, 'i'},
               {"outputdir", 1, 0 , 'o'},
               {"p4ssport", 1, 0, 'p'},
               {"p4pg", 1, 0, 'p'},
               {"p4wd", 1, 0, 'p'},
               {"help", 1, 0, 'p'},
             };
  extern char *optarg;

  MPI_Init(&argc, &argv);
  for ( c=0; c < 3; c++)  options[c]=0;
  
  
  MPI_Comm_size(MPI_COMM_WORLD, &number_of_processes);                        
  MPI_Comm_rank(MPI_COMM_WORLD, &number_of_actual_process);


  if (number_of_actual_process==MASTER_PROCESS)
  {
     if (number_of_processes<=1)
     {
       printf("\nError: it doesnt't make sense to swarm on a single processor\n");
       usage();
       MPI_Finalize();
       return(1);
     }
     /*
      * Argumentprocessing
      */
     while ((c = getopt_long (argc, argv, "c:i:o:h", long_options, &option_index)) != -1)
     {
       switch (c)
       {
       case 'c':
                 snprintf(command,FILENAME_LENGTH,"%s",optarg);
		 options[0]++;
                 break;
       case 'i':
                 snprintf(input_path,FILENAME_LENGTH,"%s",optarg);
		 options[1]++;
                 break;
       case 'o':
                snprintf(output_path,FILENAME_LENGTH,"%s",optarg);
		options[2]++;
                break;
       case 'p':
                break;
       case 'h':
       default:
              usage();
	      kill_all_workers(number_of_processes);
	      MPI_Finalize();
	      exit(1);
      }
    }
    
    for (c=0; c < 2; c++) 
    { 
       if (options[c] < 1) 
         { 
	   printf("\nError: you MUST specify at least -i inputdir and -c command\n");
	   usage(); 
	   kill_all_workers(number_of_processes); 
	   MPI_Finalize(); 
	   exit(1);
	 };
    }
    
    if ((strstr(command, OUTPUTFILE_STRING)!=NULL) && (options[2]<1) && ((ouput_dir_handle=new_directory_handle(output_path))!=NULL) )
    { 
      delete_directory_handle(ouput_dir_handle);
      printf("\nError: if you specify %s in your command \"-o outputdirectory\" must be specified as a parameter\n", OUTPUTFILE_STRING);
      usage(); 
      kill_all_workers(number_of_processes); 
      MPI_Finalize(); 
      exit(1);
    }
    ouput_dir_handle=new_directory_handle(output_path);
    if( ouput_dir_handle==NULL)
    {
      printf("\nError: output directory (%s) does not exist\n",output_path);
      usage(); 
      kill_all_workers(number_of_processes); 
      MPI_Finalize(); 
      exit(1);      
    }
    else
    {
      delete_directory_handle(ouput_dir_handle);
    };
     
    master_process(number_of_processes, number_of_actual_process);  
  }
  else
  {
     worker_process(number_of_processes, number_of_actual_process);
  }
  MPI_Finalize();
  fflush(stdout);  
  return 0;
}
